//
// LDASTools frameCPP - A library implementing the LIGO/Virgo frame
// specification
//
// Copyright (C) 2018 California Institute of Technology
//
// LDASTools frameCPP is free software; you may redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 (GPLv2) of the
// License or at your discretion, any later version.
//
// LDASTools frameCPP is distributed in the hope that it will be useful, but
// without any warranty or even the implied warranty of merchantability
// or fitness for a particular purpose. See the GNU General Public
// License (GPLv2) for more details.
//
// Neither the names of the California Institute of Technology (Caltech),
// The Massachusetts Institute of Technology (M.I.T), The Laser
// Interferometer Gravitational-Wave Observatory (LIGO), nor the names
// of its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// You should have received a copy of the licensing terms for this
// software included in the file LICENSE located in the top-level
// directory of this package. If you did not, you can view a copy at
// http://dcc.ligo.org/M1500244/LICENSE
//

#ifndef FRAMECPP__COMMON__CRC_HH
#define FRAMECPP__COMMON__CRC_HH

#define NEW_CRC 0

#if NEW_CRC
#include <stdint.h>

#include <cstdlib>

#include "ldastoolsal/reverse.hh"
#include "framecpp/Common/CRCModel.hh"

namespace FrameCPP
{
    namespace Common
    {
        template < typename MODEL >
        class CRC : public MODEL
        {
        public:
            typedef typename MODEL::crc_type crc_type;

            void SliceBy1( const void* Data, size_t Length );

            void SliceBy8( const void* Data, size_t Length );

            crc_type Finish( ) const;

        protected:
            CRC( crc_type Source );

            CRC( const CRC& Source );

        private:
            static crc_type lookup[ 8 ][ 256 ];

            crc_type crc;
        };

        template < typename MODEL >
        CRC< MODEL >::CRC( crc_type Source ) : crc( Source ^ MODEL::INITIAL )
        {
        }

        template < typename MODEL >
        CRC< MODEL >::CRC( const CRC& Source ) : crc( Source.crc )
        {
        }

        template < typename MODEL >
        inline typename CRC< MODEL >::crc_type
        CRC< MODEL >::Finish( ) const
        {
            return ( crc ^ MODEL::FINISH );
        }

        template < typename MODEL >
        inline void
        CRC< MODEL >::SliceBy1( const void* Data, size_t Length )
        {
            const uint8_t* current(
                reinterpret_cast< const uint8_t* >( Data ) );

            while ( Length-- > 0 )
            {
                crc = ( crc >> 8 ) ^ lookup[ 0 ][ ( crc & 0xFF ) ^ *current++ ];
            }
        }

        template < typename MODEL >
        inline void
        CRC< MODEL >::SliceBy8( const void* Data, size_t Length )
        {
            const crc_type* current(
                reinterpret_cast< const crc_type* >( Data ) );

            // process eight bytes at once (Slicing-by-8)
            while ( Length >= 8 )
            {
#if __BYTE_ORDER == __BIG_ENDIAN
                crc_type one = *current++ ^ bswap( crc );
                crc_type two = *current++;

                crc = lookup[ 0 ][ two & 0xFF ] ^
                    lookup[ 1 ][ ( two >> 8 ) & 0xFF ] ^
                    lookup[ 2 ][ ( two >> 16 ) & 0xFF ] ^
                    lookup[ 3 ][ ( two >> 24 ) & 0xFF ] ^
                    lookup[ 4 ][ one & 0xFF ] ^
                    lookup[ 5 ][ ( one >> 8 ) & 0xFF ] ^
                    lookup[ 6 ][ ( one >> 16 ) & 0xFF ] ^
                    lookup[ 7 ][ ( one >> 24 ) & 0xFF ];
#else
                crc_type one = *current++ ^ crc;
                crc_type two = *current++;

                crc = lookup[ 0 ][ ( two >> 24 ) & 0xFF ] ^
                    lookup[ 1 ][ ( two >> 16 ) & 0xFF ] ^
                    lookup[ 2 ][ ( two >> 8 ) & 0xFF ] ^
                    lookup[ 3 ][ two & 0xFF ] ^
                    lookup[ 4 ][ ( one >> 24 ) & 0xFF ] ^
                    lookup[ 5 ][ ( one >> 16 ) & 0xFF ] ^
                    lookup[ 6 ][ ( one >> 8 ) & 0xFF ] ^
                    lookup[ 7 ][ one & 0xFF ];
#endif

                Length -= 8;
            }

            SliceBy1( current, Length );
        }

        //-------------------------------------------------------------------
        // define some standard types
        //-------------------------------------------------------------------
        typedef class CRC< crc32_model_type > crc32_type;
    } // namespace Common
} // namespace FrameCPP

#endif /* NEW_CRC */

#endif /* FRAMECPP__COMMON__CRC_HH */
