//
// LDASTools frameCPP - A library implementing the LIGO/Virgo frame
// specification
//
// Copyright (C) 2018 California Institute of Technology
//
// LDASTools frameCPP is free software; you may redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 (GPLv2) of the
// License or at your discretion, any later version.
//
// LDASTools frameCPP is distributed in the hope that it will be useful, but
// without any warranty or even the implied warranty of merchantability
// or fitness for a particular purpose. See the GNU General Public
// License (GPLv2) for more details.
//
// Neither the names of the California Institute of Technology (Caltech),
// The Massachusetts Institute of Technology (M.I.T), The Laser
// Interferometer Gravitational-Wave Observatory (LIGO), nor the names
// of its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// You should have received a copy of the licensing terms for this
// software included in the file LICENSE located in the top-level
// directory of this package. If you did not, you can view a copy at
// http://dcc.ligo.org/M1500244/LICENSE
//

#ifndef FRAMECPP__COMMON__CRCMODEL_HH
#define FRAMECPP__COMMON__CRCMODEL_HH

#include <stdint.h>

#define GENERIC_MODEL_SIGNATURE class DT, DT POLY, DT INIT, DT FIN

namespace FrameCPP
{
    namespace Common
    {
        // DT - Data type for each element of the table
        // POYNOMIAL - Constant representing the Polynomial (MSB-first code )
        // REVERSED - Constant represetning the Polynomial (LSB-first code )
        // REVERSED_RECIPROCAL - Constant representing the Polynomial
        //     (Koopman notation)
        template < GENERIC_MODEL_SIGNATURE >
        struct CRCModel
        {
        public:
            typedef DT crc_type;

            static struct polynomial_type
            {
                static const crc_type polynomial = POLY;
                inline static crc_type
                normal( )
                {
                    return polynomial;
                }

                inline static crc_type
                reversed( )
                {
                    crc_type source( polynomial );
                    crc_type retval( source );
                    size_t   s = ( sizeof( crc_type ) * 8 ) - 1;

                    for ( source >>= 1; source; source >>= 1, --s )
                    {
                        retval <<= 1;
                        retval |= ( source & 1 );
                    }
                    retval <<= s;
                    return retval;
                }
                /* static const crc_type	reversed_reciprocal; */
            } polynomial;
            static const crc_type INITIAL = INIT;
            static const crc_type FINISH = FIN;

        private:
        };

        //-------------------------------------------------------------------
        // crc-32
        // Polynomial: 0x04C11DB7
        // Reversed: 0xEDB88320
        //-------------------------------------------------------------------
        typedef CRCModel< uint32_t, 0x04C11DB7, 0xFFFFFFFF, 0xFFFFFFFF >
            crc32_model_type;
        template <>
        struct crc32_model_type::polynomial_type crc32_model_type::polynomial;
    } // namespace Common
} // namespace FrameCPP

#endif /* FRAMECPP__COMMON__CRCMODEL_HH */
