//
// LDASTools frameCPP - A library implementing the LIGO/Virgo frame
// specification
//
// Copyright (C) 2018 California Institute of Technology
//
// LDASTools frameCPP is free software; you may redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 (GPLv2) of the
// License or at your discretion, any later version.
//
// LDASTools frameCPP is distributed in the hope that it will be useful, but
// without any warranty or even the implied warranty of merchantability
// or fitness for a particular purpose. See the GNU General Public
// License (GPLv2) for more details.
//
// Neither the names of the California Institute of Technology (Caltech),
// The Massachusetts Institute of Technology (M.I.T), The Laser
// Interferometer Gravitational-Wave Observatory (LIGO), nor the names
// of its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// You should have received a copy of the licensing terms for this
// software included in the file LICENSE located in the top-level
// directory of this package. If you did not, you can view a copy at
// http://dcc.ligo.org/M1500244/LICENSE
//

#include <framecpp_config.h>

#include <stdexcept>

#include "framecpp/Common/FrameSpec.hh"
#include "framecpp/Common/IOStream.hh"

#include "framecpp/Detectors.hh"

namespace
{
    using ::FrameCPP::FrDetector;

    //---------------------------------------------------------------------
    // GEO -
    // Values taken from lal/src/tools/LALDetectors.h
    //---------------------------------------------------------------------
    static FrDetector G1 = FrDetector( "GEO_600",
                                       "G1",
                                       0.17116780435,
                                       0.91184982752, // long, lat
                                       114.425, // elevation
                                       1.19360100484,
                                       5.83039279401, // azimuth
                                       0.00000000000,
                                       0.00000000000, // altitude
                                       300.00000000000,
                                       300.00000000000, // midpoint
                                       0 // LocalTime
    );
    //---------------------------------------------------------------------
    // HANFORD - Derived from:
    // /archive/frames/S5/L3/LHO/H-RDS_R_L3-8766/H-RDS_R_L3-876699904-256.gwf
    //---------------------------------------------------------------------
    static FrDetector H1 = FrDetector( "LHO_4k",
                                       "H1",
                                       -2.08406,
                                       0.810795, // long, lat
                                       142.554, // elevation
                                       5.65488,
                                       4.084808, // azimuth
                                       -0.0006195,
                                       0.0000125, // altitude
                                       1997.54,
                                       1997.52, // midpoint
                                       0 // LocalTime
    );
    static FrDetector H2 = FrDetector( "LHO_2k",
                                       "H2",
                                       -2.08406,
                                       0.810795, // long, lat
                                       142.554, // elevation
                                       5.65488,
                                       4.084808, // azimuth
                                       -0.0006195,
                                       0.0000125, // altitude
                                       1004.5,
                                       1004.5, // midpoint
                                       0 // LocalTime
    );
    //---------------------------------------------------------------------
    // KAGRA -
    // Values taken from lal/src/tools/LALDetectors.h
    //---------------------------------------------------------------------
    static FrDetector K1 = FrDetector( "KAGRA",
                                       "K1",
                                       2.396441015,
                                       0.6355068497, // long, lat
                                       414.181, // elevation
                                       1.054113,
                                       -0.5166798, // azimuth
                                       0.0031414,
                                       -0.0036270, // altitude
                                       1513.2535,
                                       1511.611, // midpoint
                                       0 // LocalTime
    );
    //---------------------------------------------------------------------
    // LIVINGSTON - Derived from:
    // /archive/frames/S5/L3/LLO/L-RDS_R_L3-8766/L-RDS_R_L3-876699904-256.gwf
    //---------------------------------------------------------------------
    static FrDetector L1 = FrDetector( "LLO_4k",
                                       "L1",
                                       -1.58431,
                                       0.533423, // long, lat, elevation
                                       -6.574, // elevation
                                       4.40318,
                                       2.83238, // azimuth
                                       -0.0003121,
                                       -0.0006107, // altitude
                                       1997.57,
                                       1997.57, // midpoint
                                       0 // LocalTime
    );
    //---------------------------------------------------------------------
    // TAMA -
    // Values taken from lal/src/tools/LALDetectors.h
    //---------------------------------------------------------------------
    static FrDetector T1 = FrDetector( "TAMA_300",
                                       "T1",
                                       2.43536359469,
                                       0.62267336022, // long, lat,
                                       90.0, // elevation
                                       4.71238898038,
                                       3.14159265359, // azimuth
                                       0.00000000000,
                                       0.00000000000, // altitude
                                       150.00000000000,
                                       150.00000000000, // midpoint
                                       0 // LocalTime
    );
    //---------------------------------------------------------------------
    // VIRGO - Derived from:
    // /archive/frames/WSR8/HrecV3/Virgo/V-HrecV3-855/V-HrecV3-855302400-2240.gwf
    //---------------------------------------------------------------------
    static FrDetector V1 = FrDetector( "Virgo",
                                       "V1",
                                       -10.5045,
                                       43.6316, // long, lat,
                                       51.884, // elevation
                                       -0.3229,
                                       4.3895, // azimuth
                                       0.0,
                                       0.0, // altitude
                                       1500.0,
                                       1500.0, // midpoint
                                       3600 // LocalTime
    );
} // namespace

namespace FrameCPP
{
    const FrDetector&
    GetDetector( detector_location_type Location, const GPSTime& Time )
    {
        switch ( Location )
        {
        case DETECTOR_LOCATION_G1:
            return G1;
        case DETECTOR_LOCATION_H1:
            return H1;
        case DETECTOR_LOCATION_H2:
            return H2;
        case DETECTOR_LOCATION_K1:
            return K1;
        case DETECTOR_LOCATION_L1:
            return L1;
        case DETECTOR_LOCATION_T1:
            return T1;
        case DETECTOR_LOCATION_V1:
            return V1;
        default:
            break;
        }
        throw std::range_error( "Unknown detector requested" );
    }
} // namespace FrameCPP
