//
// LDASTools frameCPP - A library implementing the LIGO/Virgo frame
// specification
//
// Copyright (C) 2018 California Institute of Technology
//
// LDASTools frameCPP is free software; you may redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 (GPLv2) of the
// License or at your discretion, any later version.
//
// LDASTools frameCPP is distributed in the hope that it will be useful, but
// without any warranty or even the implied warranty of merchantability
// or fitness for a particular purpose. See the GNU General Public
// License (GPLv2) for more details.
//
// Neither the names of the California Institute of Technology (Caltech),
// The Massachusetts Institute of Technology (M.I.T), The Laser
// Interferometer Gravitational-Wave Observatory (LIGO), nor the names
// of its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// You should have received a copy of the licensing terms for this
// software included in the file LICENSE located in the top-level
// directory of this package. If you did not, you can view a copy at
// http://dcc.ligo.org/M1500244/LICENSE
//

#ifndef FrameCPP_VERSION_8_FrDetector_HH
#define FrameCPP_VERSION_8_FrDetector_HH

#include "framecpp/Version8/impl/FrDetectorClassicIO.hh"

#if !defined( SWIGIMPORTED )
namespace FrameCPP
{
    namespace Version_8
    {
        //===================================================================
        /// \brief Detector data
        //===================================================================
        class FrDetector : public FrDetectorImpl::ClassicIO< FrDetector >
        {
        public:
            //-----------------------------------------------------------------
            /// \brief Default constructor
            ///
            /// \return
            ///     A new instance of an FrDetector object
            //-----------------------------------------------------------------
            FrDetector( );

            //-----------------------------------------------------------------
            /// \brief Constructor
            ///
            /// \param[in] Name
            ///     Instrument name
            /// \param[in] Prefix
            ///     Channel prefix for this detector
            /// \param[in] Longitude
            ///     Detector vertex longitude
            /// \param[in] Latitude
            ///     Detector vertex latitutde.
            /// \param[in] Elevation
            ///     Vertex elevation.
            /// \param[in] ArmXazimuth
            ///     Orientation of X arm.
            /// \param[in] ArmYazimuth
            ///     Orientation of Y arm.
            /// \param[in] ArmXaltitude
            ///     Altidue angle of X arm.
            /// \param[in] ArmYaltitude
            ///     Altidue angle of Y arm.
            /// \param[in] ArmXmidpoint
            ///     Distance between the detector vertex and the middle of
            ///     the X cavity.
            /// \param[in] ArmYmidpoint
            ///     Distance between the detector vertex and the middle of
            ///     the Y cavity.
            /// \param[in] LocalTime
            ///     Local seasonal time.
            ///
            /// \return
            ///     A new instance of an FrDetector object
            //-----------------------------------------------------------------
            FrDetector( const name_type&        Name,
                        const char*             Prefix,
                        const longitude_type    Longitude,
                        const latitude_type     Latitude,
                        const elevation_type    Elevation,
                        const armXazimuth_type  ArmXazimuth,
                        const armYazimuth_type  ArmYazimuth,
                        const armXaltitude_type ArmXaltitude,
                        const armYaltitude_type ArmYaltitude,
                        const armXmidpoint_type ArmXmidpoint,
                        const armYmidpoint_type ArmYmidpoint,
                        const localTime_type    LocalTime );

            //-----------------------------------------------------------------
            /// \brief Copy constructor
            ///
            /// \param[in] detector
            ///     The source instance to copy
            ///
            /// \return
            ///     A new instance of an FrDetector object
            //-----------------------------------------------------------------
            FrDetector( const FrDetector& detector );

            const std::string& GetName( ) const;

            //-----------------------------------------------------------------
            /// \brief Promotion Constructor
            ///
            /// \param[in] Source
            ///     An FrDetector structure from a previous frame specification.
            /// \param[in] Stream
            ///     The stream from which the earliest version of the
            ///     FrDetector structure was read.
            ///
            /// \return
            ///     A new instance of an FrDetector object
            //-----------------------------------------------------------------
            FrDetector( Previous::FrDetector& Source, istream_type* Stream );

            //-----------------------------------------------------------------
            /// \brief Merge with another FrDetector
            ///
            /// \param[in] RHS
            ///     The source of the information to append to this FrDetector
            ///     structure.
            ///
            /// \return
            ///     A reference to this object
            //-----------------------------------------------------------------
            FrDetector& Merge( const FrDetector& RHS );

            //-----------------------------------------------------------------
            /// \brief equality operator
            ///
            /// \param[in] RHS
            ///     The FrDetector object to be compared.
            ///
            /// \return
            ///     The value true is returned if the object is equivelent
            ///     to the RHS instance; false otherwise.
            //-----------------------------------------------------------------
            bool operator==( const FrDetector& RHS ) const;

            //-----------------------------------------------------------------
            /// \brief equality operator for abstract data type
            ///
            /// \param[in] RHS
            ///     The object to be compared.
            ///
            /// \return
            ///     The value true is returned if this object is equivelent
            ///     to the RHS instance; false otherwise.
            //-----------------------------------------------------------------
            virtual bool
            operator==( const Common::FrameSpec::Object& RHS ) const;

        private:
            //-----------------------------------------------------------------
            //-----------------------------------------------------------------
            typedef ObjectWithChecksum< Common::FrDetector > object_type;

            using Common::FrameSpec::Object::Create;

            FrDetector( istream_type& Stream );
        };

        inline bool
        FrDetector::operator==( const FrDetector& RHS ) const
        {
            return ( ( this == &RHS ) ||
                     ( (Data&)( *this ) == (Data&)( RHS ) ) );
        }
    } // namespace Version_8
} // namespace FrameCPP
#endif /* ! defined(SWIGIMPORTED) */
#endif /* FrameCPP_VERSION_8_FrDetector_HH */
