//
// LDASTools frameCPP - A library implementing the LIGO/Virgo frame
// specification
//
// Copyright (C) 2019 California Institute of Technology
//
// LDASTools frameCPP is free software; you may redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 (GPLv2) of the
// License or at your discretion, any later version.
//
// LDASTools frameCPP is distributed in the hope that it will be useful, but
// without any warranty or even the implied warranty of merchantability
// or fitness for a particular purpose. See the GNU General Public
// License (GPLv2) for more details.
//
// Neither the names of the California Institute of Technology (Caltech),
// The Massachusetts Institute of Technology (M.I.T), The Laser
// Interferometer Gravitational-Wave Observatory (LIGO), nor the names
// of its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// You should have received a copy of the licensing terms for this
// software included in the file LICENSE located in the top-level
// directory of this package. If you did not, you can view a copy at
// http://dcc.ligo.org/M1500244/LICENSE
//

#ifndef FrameCPP_VERSION_8__IMPL__FrMsgData_HH
#define FrameCPP_VERSION_8__IMPL__FrMsgData_HH

#include "framecpp/Version8/impl/FrObjectMetaData.hh"

#include "framecpp/Version8/STRING.hh"
#include "framecpp/Version8/GPSTime.hh"

namespace FrameCPP
{
    namespace Version_8
    {
        namespace FrMsgImpl
        {
            class Data : public Impl::FrObjectMetaData< Data >
            {
            public:
                typedef std::string alarm_type;
                typedef std::string message_type;
                typedef INT_4U      severity_type;
                typedef GPSTime     gtime_type;

                //-----------------------------------------------------------------
                /// \brief Retrieve the name of message, error flag, or alarm
                /// state.
                ///
                /// \return
                ///     The name of message, error flag, or alarm state.
                //-----------------------------------------------------------------
                const alarm_type&
                GetAlarm( ) const
                {
                    return ( alarm );
                }

                //-----------------------------------------------------------------
                /// \brief Retieve the message body.
                ///
                /// \return
                ///     The message body.
                //-----------------------------------------------------------------
                const message_type&
                GetMessage( ) const
                {
                    return message;
                }

                //-----------------------------------------------------------------
                /// \brief Retieve the message severity level.
                ///
                /// \return
                ///     The message severity level.
                //-----------------------------------------------------------------
                severity_type
                GetSeverity( ) const
                {
                    return severity;
                }

                //-----------------------------------------------------------------
                /// \brief Retrieve the GPS time corresponding to this FrMsg.
                ///
                /// \return
                ///     The GPS time corresponding to this FrMsg.
                //-----------------------------------------------------------------
                const gtime_type&
                GetGTime( ) const
                {
                    return GTime;
                }

                //---------------------------------------------------------------
                /// \brief equality operator
                ///
                /// \param[in] RHS
                ///     The fr_msg_data_type object to be compared.
                ///
                /// \return
                ///     The value true is returned if this object is
                ///     equivelent to the RHS instance; false otherwise.
                //---------------------------------------------------------------
                bool
                operator==( const Data& RHS ) const
                {
                    return ( ( alarm == RHS.alarm ) &&
                             ( message == RHS.message ) &&
                             ( severity == RHS.severity ) &&
                             ( GTime == RHS.GTime ) );
                }

            protected:
                typedef STRING internal_alarm_type;
                typedef STRING internal_message_type;
                //---------------------------------------------------------------
                /// Name o message, error flag or alarm state.
                //---------------------------------------------------------------
                alarm_type alarm;
                //---------------------------------------------------------------
                /// Message body.
                //---------------------------------------------------------------
                message_type message;
                //---------------------------------------------------------------
                /// Message severity level.
                /// The default level is zero.
                /// \note
                ///     These are yet to be defined.
                //---------------------------------------------------------------
                severity_type severity;
                //---------------------------------------------------------------
                /// The GPS time corresponding alarm.
                //---------------------------------------------------------------
                gtime_type GTime;
            };
        } // namespace FrMsgImpl
    } // namespace Version_8
} // namespace FrameCPP

FR_OBJECT_META_DATA_DECLARE( FrMsgImpl )

#endif /* FrameCPP_VERSION_8__IMPL__FrMsgData_HH */
