//
// LDASTools frameCPP - A library implementing the LIGO/Virgo frame
// specification
//
// Copyright (C) 2019 California Institute of Technology
//
// LDASTools frameCPP is free software; you may redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 (GPLv2) of the
// License or at your discretion, any later version.
//
// LDASTools frameCPP is distributed in the hope that it will be useful, but
// without any warranty or even the implied warranty of merchantability
// or fitness for a particular purpose. See the GNU General Public
// License (GPLv2) for more details.
//
// Neither the names of the California Institute of Technology (Caltech),
// The Massachusetts Institute of Technology (M.I.T), The Laser
// Interferometer Gravitational-Wave Observatory (LIGO), nor the names
// of its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// You should have received a copy of the licensing terms for this
// software included in the file LICENSE located in the top-level
// directory of this package. If you did not, you can view a copy at
// http://dcc.ligo.org/M1500244/LICENSE
//

#ifndef FrameCPP_VERSION_8__IMPL__FrameHData_HH
#define FrameCPP_VERSION_8__IMPL__FrameHData_HH

#if defined( __cplusplus )
#if !defined( SWIG )
#include <memory>
#include <vector>

#include <boost/shared_ptr.hpp>

#include "ldastoolsal/types.hh"

#include "framecpp/Common/Container.hh"

#endif /* ! defined(SWIG) */
#endif /* defined(__cplusplus) */

#include "framecpp/Version7/FrameH.hh"

#if !defined( SWIGIMPORTED )
#include "framecpp/Version8/impl/FrObjectMetaData.hh"

#include "framecpp/Version8/GPSTime.hh"
#include "framecpp/Version8/STRING.hh"
#include "framecpp/Version8/PTR_STRUCT.hh"
#include "framecpp/Version8/FrSE.hh"

#include "framecpp/Version8/FrDetector.hh"
#include "framecpp/Version8/FrEvent.hh"
#include "framecpp/Version8/FrHistory.hh"
#include "framecpp/Version8/FrProcData.hh"
#include "framecpp/Version8/FrRawData.hh"
#include "framecpp/Version8/FrSimData.hh"
#include "framecpp/Version8/FrSimEvent.hh"
#include "framecpp/Version8/FrSummary.hh"
#include "framecpp/Version8/FrTable.hh"
#include "framecpp/Version8/FrVect.hh"
#endif /* !defined(SWIGIMPORTED) */

#undef FRAME_H_NAME_TYPE
#define FRAME_H_NAME_TYPE std::string

#if defined( __cplusplus )
#if !defined( SWIG )

namespace FrameCPP
{
    namespace Version_8
    {
        namespace FrameHImpl
        {
            class Data : public Impl::FrObjectMetaData< Data >
            {
            public:
                typedef FRAME_H_NAME_TYPE         name_type;
                typedef FRAME_H_RUN_TYPE          run_type;
                typedef FRAME_H_FRAME_TYPE        frame_type;
                typedef FRAME_H_DATA_QUALITY_TYPE dataQuality_type;
                typedef FRAME_H_G_TIME_S_TYPE     GTimeS_type;
                typedef FRAME_H_G_TIME_N_TYPE     GTimeN_type;
                typedef FRAME_H_G_TIME_TYPE       GTime_type;
                typedef FRAME_H_U_LEAP_S_TYPE     ULeapS_type;
                typedef FRAME_H_LOCAL_TIME_TYPE   localTime_type;
                typedef FRAME_H_DT_TYPE           dt_type;
                //-------------------------------------------------------
                static const INT_2U                    MAX_REF = 13;
                typedef boost::shared_ptr< FrRawData > rawData_type;

                typedef Common::SearchContainer< FrVect, &FrVect::GetName >
                                                       type_type;
                typedef type_type::iterator            type_iterator;
                typedef type_type::const_iterator      const_type_iterator;
                typedef type_type::hash_iterator       type_hash_iterator;
                typedef type_type::const_hash_iterator const_type_hash_iterator;

                typedef Common::SearchContainer< FrVect, &FrVect::GetName >
                                                       user_type;
                typedef user_type::iterator            user_iterator;
                typedef user_type::const_iterator      const_user_iterator;
                typedef user_type::hash_iterator       user_hash_iterator;
                typedef user_type::const_hash_iterator const_user_hash_iterator;

                typedef Common::SearchContainer< FrDetector,
                                                 &FrDetector::GetName >
                                                       detectSim_type;
                typedef detectSim_type::iterator       detectSim_iterator;
                typedef detectSim_type::const_iterator const_detectSim_iterator;
                typedef detectSim_type::hash_iterator  detectSim_hash_iterator;
                typedef detectSim_type::const_hash_iterator
                    const_detectSim_hash_iterator;

                typedef Common::SearchContainer< FrDetector,
                                                 &FrDetector::GetName >
                                                  detectProc_type;
                typedef detectProc_type::iterator detectProc_iterator;
                typedef detectProc_type::const_iterator
                                                       const_detectProc_iterator;
                typedef detectProc_type::hash_iterator detectProc_hash_iterator;
                typedef detectProc_type::const_hash_iterator
                    const_detectProc_hash_iterator;

                typedef Common::SearchContainer< FrEvent,
                                                 &FrEvent::GetNameSlow >
                                                   event_type;
                typedef event_type::iterator       event_iterator;
                typedef event_type::const_iterator const_event_iterator;
                typedef event_type::hash_iterator  event_hash_iterator;
                typedef event_type::const_hash_iterator
                    const_event_hash_iterator;

                typedef Common::SearchContainer< FrHistory,
                                                 &FrHistory::GetName >
                                                     history_type;
                typedef history_type::iterator       history_iterator;
                typedef history_type::const_iterator const_history_iterator;
                typedef history_type::hash_iterator  history_hash_iterator;
                typedef history_type::const_hash_iterator
                    const_history_hash_iterator;

                typedef Common::SearchContainer< FrProcData,
                                                 &FrProcData::GetNameSlow >
                                                      procData_type;
                typedef procData_type::iterator       procData_iterator;
                typedef procData_type::const_iterator const_procData_iterator;
                typedef procData_type::hash_iterator  procData_hash_iterator;
                typedef procData_type::const_hash_iterator
                    const_procData_hash_iterator;

                typedef Common::SearchContainer< FrSimData,
                                                 &FrSimData::GetNameSlow >
                                                     simData_type;
                typedef simData_type::iterator       simData_iterator;
                typedef simData_type::const_iterator const_simData_iterator;
                typedef simData_type::hash_iterator  simData_hash_iterator;
                typedef simData_type::const_hash_iterator
                    const_simData_hash_iterator;

                typedef Common::SearchContainer< FrSimEvent,
                                                 &FrSimEvent::GetNameSlow >
                                                      simEvent_type;
                typedef simEvent_type::iterator       simEvent_iterator;
                typedef simEvent_type::const_iterator const_simEvent_iterator;
                typedef simEvent_type::hash_iterator  simEvent_hash_iterator;
                typedef simEvent_type::const_hash_iterator
                    const_simEvent_hash_iterator;

                typedef Common::SearchContainer< FrSummary,
                                                 &FrSummary::GetName >
                                                   summaryData_type;
                typedef summaryData_type::iterator summaryData_iterator;
                typedef summaryData_type::const_iterator
                    const_summaryData_iterator;
                typedef summaryData_type::hash_iterator
                    summaryData_hash_iterator;
                typedef summaryData_type::const_hash_iterator
                    const_summaryData_hash_iterator;

                typedef Common::SearchContainer< FrVect, &FrVect::GetName >
                                                     auxData_type;
                typedef auxData_type::iterator       auxData_iterator;
                typedef auxData_type::const_iterator const_auxData_iterator;
                typedef auxData_type::hash_iterator  auxData_hash_iterator;
                typedef auxData_type::const_hash_iterator
                    const_auxData_hash_iterator;

                typedef Common::SearchContainer< FrTable,
                                                 &FrTable::GetNameSlow >
                                                      auxTable_type;
                typedef auxTable_type::iterator       auxTable_iterator;
                typedef auxTable_type::const_iterator const_auxTable_iterator;
                typedef auxTable_type::hash_iterator  auxTable_hash_iterator;
                typedef auxTable_type::const_hash_iterator
                    const_auxTable_hash_iterator;
                //-------------------------------------------------------

#if 0
                //-----------------------------------------------------------------
                /// \brief Default constructor
                ///
                /// Construct a new FrameHNPS object with default values
                ///
                /// \return
                ///     A new FrameHNPS object.
                //-----------------------------------------------------------------
                Data( );

                //-----------------------------------------------------------------
                /// \brief Constructor
                ///
                /// Construct a new FrameHNPS object with explicate values.
                ///
                /// \param[in] name
                ///     Name of project or other experiment description.
                /// \param[in] run
                ///     Run number (number < 0 reserved fro simulated data);
                ///     monotonic for experimental runs.
                /// \param[in] frame
                ///     Frame number, monotonically increasing until end of run,
                ///     re-starting from 0 with each new run.
                /// \param[in] time
                ///     Frame start time.
                /// \param[in] ULeapS
                ///     The integer number of leap seconds between GPS/TAI
                ///     and UTC in epoch when the frame is written
                /// \param[in] dt
                ///     Frame length in seconds.
                /// \param[in] dqual
                ///     A logical 32-bit word to denote top level quality of
                ///     data. Lowest order bits are reserved in pairs for the
                ///     various GW detectors.
                ///
                /// \return
                ///     A new FrameHNPS object.
                //-----------------------------------------------------------------
                Data( const name_type&  name,
                      run_type          run,
                      frame_type        frame,
                      const GTime_type& time,
                      ULeapS_type       ULeapS,
                      const dt_type     dt,
                      dataQuality_type  dqual );
#endif /* 0 */

                inline rawData_type*
                AddressOfRawData( ) const
                {
                    return ( const_cast< rawData_type* >( &rawData ) );
                }

                //-----------------------------------------------------------------
                /// \brief Retrieve name of project
                ///
                /// \return
                ///     Name of project
                //-----------------------------------------------------------------
                const name_type& GetName( ) const;

                //-----------------------------------------------------------------
                /// \brief Run number
                ///
                /// \return
                ///     The run number of the frame
                //-----------------------------------------------------------------
                run_type GetRun( ) const;

                //-----------------------------------------------------------------
                /// \brief Frame number
                ///
                /// \return
                ///     The frame number of the frame
                //-----------------------------------------------------------------
                frame_type GetFrame( ) const;

                //-----------------------------------------------------------------
                /// \brief Data quality flag
                ///
                /// \return
                ///     The data quality flag for the frame
                //-----------------------------------------------------------------
                dataQuality_type GetDataQuality( ) const;

                //-----------------------------------------------------------------
                /// \brief Start Time
                ///
                /// \return
                ///     The start time of the frame
                //-----------------------------------------------------------------
                const GTime_type& GetGTime( ) const;

                //-----------------------------------------------------------------
                /// \brief Leap seconds
                ///
                /// \return
                ///     The integer number of leap seconds associated with
                ///     the start time.
                //-----------------------------------------------------------------
                ULeapS_type GetULeapS( ) const;

                //-----------------------------------------------------------------
                /// \brief Frame length
                ///
                /// \return
                ///     Length of frame in seconds.
                //-----------------------------------------------------------------
                dt_type GetDt( ) const;

                //-----------------------------------------------------------------
                /// \brief Establish the name
                ///
                /// \param[in] Name
                ///     The name to associate with the frame
                //-----------------------------------------------------------------
                void SetName( const name_type& Name );

                //-----------------------------------------------------------------
                /// \brief Establish the run number
                ///
                /// \param[in] Run
                ///     The run number to associate with the frame.
                //-----------------------------------------------------------------
                void SetRun( run_type Run );

                //-----------------------------------------------------------------
                /// \brief Establish the data quality flag
                ///
                /// \param[in] DataQuality
                ///     The value of the data quality flag.
                //-----------------------------------------------------------------
                void SetDataQuality( dataQuality_type DataQuality );

                //-----------------------------------------------------------------
                /// \brief Establish the frame duration.
                ///
                /// \param[in] Dt
                ///     The length of the frame in seconds
                //-----------------------------------------------------------------
                void SetDt( dt_type Dt );

                //-----------------------------------------------------------------
                /// \brief Establish the start time.
                ///
                /// \param[in] GTime
                ///     The start time of the frame.
                //-----------------------------------------------------------------
                void SetGTime( const GTime_type& GTime );

                //-----------------------------------------------------------------
                /// \brief Establish leap second offset.
                ///
                /// \param[in] Value
                ///     Number of leap seconds since GPS epoc.
                //-----------------------------------------------------------------
                void SetULeapS( ULeapS_type Value );

                //-----------------------------------------------------------------
                /// \brief Retrieve the raw data reference
                ///
                /// \return
                ///     A pointer to the raw data structure.
                //-----------------------------------------------------------------
                inline rawData_type
                GetRawData( ) const
                {
                    return rawData;
                }

                //-----------------------------------------------------------------
                /// \brief Retrieve the raw data reference
                ///
                /// \return
                ///     A pointer to the raw data structure.
                //-----------------------------------------------------------------
                inline rawData_type
                GetRawData( )
                {
                    return rawData;
                }

                inline const type_type&
                RefType( ) const
                {
                    return type;
                }

                inline type_type&
                RefType( )
                {
                    return type;
                }

                inline const user_type&
                RefUser( ) const
                {
                    return user;
                }

                inline user_type&
                RefUser( )
                {
                    return user;
                }

                inline const detectSim_type&
                RefDetectSim( ) const
                {
                    return detectSim;
                }

                inline detectSim_type&
                RefDetectSim( )
                {
                    return detectSim;
                }

                inline const detectProc_type&
                RefDetectProc( ) const
                {
                    return detectProc;
                }

                inline detectProc_type&
                RefDetectProc( )
                {
                    return detectProc;
                }

                inline const history_type&
                RefHistory( ) const
                {
                    return history;
                }

                inline history_type&
                RefHistory( )
                {
                    return history;
                }

                inline const procData_type&
                RefProcData( ) const
                {
                    return procData;
                }

                inline procData_type&
                RefProcData( )
                {
                    return procData;
                }

                inline const simData_type&
                RefSimData( ) const
                {
                    return simData;
                }

                inline simData_type&
                RefSimData( )
                {
                    return simData;
                }

                inline const event_type&
                RefEvent( ) const
                {
                    return event;
                }

                inline event_type&
                RefEvent( )
                {
                    return event;
                }

                inline const simEvent_type&
                RefSimEvent( ) const
                {
                    return simEvent;
                }

                inline simEvent_type&
                RefSimEvent( )
                {
                    return simEvent;
                }

                inline const summaryData_type&
                RefSummaryData( ) const
                {
                    return summaryData;
                }

                inline summaryData_type&
                RefSummaryData( )
                {
                    return summaryData;
                }

                inline const auxData_type&
                RefAuxData( ) const
                {
                    return auxData;
                }

                inline auxData_type&
                RefAuxData( )
                {
                    return auxData;
                }

                inline const auxTable_type&
                RefAuxTable( ) const
                {
                    return auxTable;
                }

                inline auxTable_type&
                RefAuxTable( )
                {
                    return auxTable;
                }

                inline void
                SetRawData( rawData_type r )
                {
                    rawData = r;
                }

                inline bool
                operator==( const Data& RHS ) const
                {
                    return ( ( this == &RHS ) ||
                             ( ( name.compare( RHS.name ) == 0 ) &&
                               ( run == RHS.run ) && ( frame == RHS.frame ) &&
                               ( dataQuality == RHS.dataQuality ) &&
                               ( GTime == RHS.GTime ) &&
                               ( ULeapS == RHS.ULeapS ) && ( dt == RHS.dt ) ) );
                }

            protected:
                //-------------------------------------------------------
                name_type        name;
                run_type         run;
                frame_type       frame;
                dataQuality_type dataQuality;
                GTime_type       GTime;
                ULeapS_type      ULeapS;
                dt_type          dt;
                //-------------------------------------------------------
                type_type            type;
                user_type            user;
                detectSim_type       detectSim;
                detectProc_type      detectProc;
                mutable history_type history;
                rawData_type         rawData;
                procData_type        procData;
                simData_type         simData;
                event_type           event;
                simEvent_type        simEvent;
                summaryData_type     summaryData;
                auxData_type         auxData;
                auxTable_type        auxTable;
            };

            //-------------------------------------------------------------------
            /// Retrieve the name of project or other experiment description.
            //-------------------------------------------------------------------
            inline const Data::name_type&
            Data::GetName( ) const
            {
                return name;
            }

            //-------------------------------------------------------------------
            /// Retrieve the run number associated with the frame.
            //-------------------------------------------------------------------
            inline Data::run_type
            Data::GetRun( ) const
            {
                return run;
            }

            //-------------------------------------------------------------------
            /// Retrieve the frame number associated with the frame
            //-------------------------------------------------------------------
            inline Data::frame_type
            Data::GetFrame( ) const
            {
                return frame;
            }

            //-------------------------------------------------------------------
            /// Retrieve the data quality flag associated with the frame
            //-------------------------------------------------------------------
            inline Data::dataQuality_type
            Data::GetDataQuality( ) const
            {
                return dataQuality;
            }

            //-------------------------------------------------------------------
            /// Retrieve the start time of the frame.
            //-------------------------------------------------------------------
            inline const GPSTime&
            Data::GetGTime( ) const
            {
                return GTime;
            }

            //-------------------------------------------------------------------
            /// Retrieve the number of leap seconds associated with the frame.
            //-------------------------------------------------------------------
            inline Data::ULeapS_type
            Data::GetULeapS( ) const
            {
                return ULeapS;
            }

            //-------------------------------------------------------------------
            /// Retrieve the duration of the frame in seconds.
            //-------------------------------------------------------------------
            inline Data::dt_type
            Data::GetDt( ) const
            {
                return dt;
            }

            //-------------------------------------------------------------------
            /// Set the project name of the frame.
            /// Values less than zero are reserved for simulated data.
            /// For experimental runs, this value is monotonicly increasing.
            //-------------------------------------------------------------------
            inline void
            Data::SetName( const std::string& Name )
            {
                name = Name;
            }

            //-------------------------------------------------------------------
            /// Set the run number of the frame.
            /// Values less than zero are reserved for simulated data.
            /// For experimental runs, this value is monotonicly increasing.
            //-------------------------------------------------------------------
            inline void
            Data::SetRun( run_type Run )
            {
                run = Run;
            }

            //-------------------------------------------------------------------
            /// Set the data quality flag of the frame.
            //-------------------------------------------------------------------
            inline void
            Data::SetDataQuality( dataQuality_type DataQuality )
            {
                dataQuality = DataQuality;
            }

            //-------------------------------------------------------------------
            /// Set the length of the frame in seconds.
            //-------------------------------------------------------------------
            inline void
            Data::SetDt( dt_type Dt )
            {
                dt = Dt;
            }

            //-------------------------------------------------------------------
            /// Set the start time of the frame.
            //-------------------------------------------------------------------
            inline void
            Data::SetGTime( const GTime_type& Value )
            {
                GTime = Value;
                ULeapS = GTime.GetLeapSeconds( );
            }

            //-------------------------------------------------------------------
            /// Set the number of leap seconds since GPS epoc.
            //-------------------------------------------------------------------
            inline void
            Data::SetULeapS( ULeapS_type Value )
            {
                ULeapS = Value;
            }

        } // namespace FrameHImpl
    } // namespace Version_8
} // namespace FrameCPP

FR_OBJECT_META_DATA_DECLARE( FrameHImpl )

#endif /* ! defined(SWIG) */
#endif /* defined(__cplusplus) */

#endif /* FrameCPP_VERSION_8__IMPL__FrameHData_HH */
