/*----------------------------------------------------------------------
   File: commData3.h
----------------------------------------------------------------------*/

#ifndef __COMMDATA3_H__
#define __COMMDATA3_H__

#include "util/fixed_width_types.h"
#include "util/timing.h"

///	\file commData3.h
///	\brief Header file with IPC communications structures
///

///	Defines max number of IPC allowed per network type
#define MAX_IPC 1024
///	Defines max number of IPC allowed on a RFM network if using DMA
#define MAX_IPC_RFM 1024
/// The number of data blocks buffered per IPC channel
#define IPC_BLOCKS 64
#define PCIE_SEG_SIZE 16

/// If one or more IPC sends take longer that this value a warning
/// will be generated (at 1Hz)
#define IPC_MAX_SEND_TIME_NS 5000

typedef struct TIMING_SIGNAL
{
    unsigned long gps_time;
    unsigned long cycle;
} TIMING_SIGNAL;

///	Struct for a single IPC xmission
typedef struct CDS_IPC_XMIT
{
    ///	Signal value being xmitted
    double data;
    ///	Combination of GDS seconds and cycle count
    unsigned long timestamp;
} CDS_IPC_XMIT;

///	Defines the array buffer in memory for IPC comms
typedef struct CDS_IPC_COMMS
{
    CDS_IPC_XMIT dBlock[ IPC_BLOCKS ][ MAX_IPC ];
} CDS_IPC_COMMS;

///	Structure to maintain all IPC information
typedef struct CDS_IPC_INFO
{
    ///	Data value to be sent or being received
    double data;
    ///	Not Used
    int sendNode;
    ///	Communication mechanism (as defined ISHME, IRFM0, IRFM1,IPCIE)
    int netType;
    ///	Cycle count to be sent as part of the timestamp data
    int sendCycle;
    ///	Rate at which data is being sent
    int sendRate;
    ///	Rate at which data is to be received
    //int rcvRate;
    ///	How many cycles (at the IPC_MAX_RATE rate) to wait between read attempts
    int rcvCycle65k;
    /// IPC number given from the IPC configuration file
    int ipcNum;
    ///	Should code send or receive this IPC
    int mode;
    ///	Errors/sec detected for a single IPC
    int errFlag;
    ///	Marks error to IPC status by network type
    int errTotal;
    ///	Name of the IPC signal from the user model
    char* name;
    ///	Name of the model which contains the IPC sender part
    char*         senderModelName;
    ///	Pointer to the IPC data memory location
    volatile CDS_IPC_COMMS* pIpcDataRead;
    volatile CDS_IPC_COMMS* pIpcDataWrite;
} CDS_IPC_INFO;

typedef struct CDS_IPC_STATS_t
{
    //
    //IPC Recv Stats
    //
    int32_t numWaitsInSec;
    int32_t ipcMaxWait_ns;
    int32_t slowIpcListIndex;

    //
    //IPC Sending Stats
    //

    //Total time to send IPCs, for all cycles in second
    uint64_t ipcTotalSendTime_ns;
    //Max time to send all IPCs, for all cycles in second
    int32_t ipcMaxSendTime_ns;

} CDS_IPC_STATS_t;

/// Indicates data is to be sent
#define ISND 1
/// Indicates data is to be received
#define IRCV 0
#define ISHME 0
#define IPCIE 1
#define IRFM0 2
#define IRFM1 3
#define IPC_BUFFER_SIZE sizeof( struct CDS_IPC_COMMS )
#define IPC_BASE_OFFSET 0x0
#define IPC_PCIE_TIME_OFFSET 0x80
#define IPC_PCIE_BASE_OFFSET 0x100
#define IPC_TOTAL_ALLOC_SIZE                                                   \
    ( IPC_PCIE_BASE_OFFSET + ( 3 * sizeof( CDS_IPC_COMMS ) ) )
#define RFM0_OFFSET ( ( 1 * sizeof( CDS_IPC_COMMS ) ) )
#define RFM1_OFFSET ( ( 2 * sizeof( CDS_IPC_COMMS ) ) )
#define IPC_PCIE_READ 2
#define IPC_PCIE_WRITE 3
#define IPC_MAX_RATE 65536
#define IPC_RFM_BLOCK_SIZE ( sizeof( struct CDS_IPC_XMIT ) * MAX_IPC )
#define IPC_RFM_XFER_SIZE ( sizeof( struct CDS_IPC_XMIT ) * MAX_IPC_RFM )

#ifdef __cplusplus
extern "C" {
#endif

typedef struct CDS_EPICS CDS_EPICS;

void commData3Init(int connects, int model_rate, int send_rate, CDS_IPC_INFO ipcInfo[] );
void commData3GetIpcNums(int totalIPCs, CDS_IPC_INFO ipcInfo[], int * numSenders, int * numReceiver);
void commData3Send(int connects, CDS_IPC_INFO ipcInfo[], int timeSec, int cycle);
int commData3Receive(int connects, CDS_IPC_INFO ipcInfo[], int timeSec, int cycle);

void commData3SetCycleMax(unsigned measured_max_us);
void commData3SetCycleStartTime(LIGO_TIMER_t cycle_start_tsc);
int commData3GetInitStatus( void );
void commData3GetStats( CDS_IPC_STATS_t * stats_ptr);

void ipc_subsys_1Hz_stats_update( volatile CDS_EPICS* epics_ptr );
void ipc_subsys_diag_reset( volatile CDS_EPICS* epics_ptr );


#ifdef __cplusplus
}
#endif


#endif // __COMMDATA3_H__
