#include "drv/iop_adc_functions.h"


#include "drv/gsc16ai64.h"
#include "drv/gsc18ai32.h"
#include "drv/gsc18ai64.h"
#include "drv/gsc_adc_common.h"
#include "drv/symmetricomGps.h"
#include "drv/spectracomGPS.h"
#include "drv/ligoPcieTiming.h"
#include "drv/gsc18ao8.h"
#include "drv/plx_9056.h"
#include "util/timing.h"
#include "fm10Gen.h"//iir_filter_biquad()
#include "controller.h" //cdsPciModules, etc
#include "rts-logger.h"
#include <linux/delay.h>

#ifdef XMIT_DOLPHIN_TIME
#include "../fe/controllerIop.h" //pcieTimer
#endif

#ifdef XMIT_DOLPHIN_TIME
#include <asm/cacheflush.h>
#endif

//
// Global Data
//

// A state variable for setting up time_shift
// When ADC_READ_PREP_TIME_SHIFT, we do a full read, but setup DMA for the next read
// When ADC_READ_ADJUST_TIME_SHIFT, We read in less data, completing the timeshift
// When ADC_READ_RUNNING, we're reading full, time_shifted data.
// State starts out ADC_READ_RUNNING when initializing, so that 1PPS check is initially
// unaffected by time_shift
// The state is set to ADC_READ_PREP_TIME_SHIFT at the moment the IOP starts with a call to iop_adc_setup_timeshift()
// Each cycle brings the state down from ADC_READ_PREP_TIME_SHIFT to ADC_READ_RUNNING and remains there.
#define ADC_READ_PREP_TIME_SHIFT 2
#define ADC_READ_ADJUST_TIME_SHIFT 1
#define ADC_READ_RUNNING 0
static int adc_read_state = ADC_READ_RUNNING;

/// After the ADCs are initialized, but before the first sample is read
/// Start the time_shift process
void
iop_adc_setup_timeshift(void) {
  adc_read_state = ADC_READ_PREP_TIME_SHIFT;
}

/// Get the DMA size needed for the time_shift cycle for a particular ADC
int
iop_adc_timeshift_dma_size(int adc)
{
  int dma_bytes = 0;
  int samples=0;
  switch ( cdsPciModules.adcType[ adc ] )
  {
    case GSC_18AI32SSC1M:
    case GSC_18AI64SSC:
      if(cdsPciModules.adcTimeShift[ adc ] < UNDERSAMPLE)
      {
        samples = cdsPciModules.adcChannels[ adc ]
                  * (UNDERSAMPLE-cdsPciModules.adcTimeShift[adc]);
      }
      else
      {
        samples = cdsPciModules.adcChannels[ adc ] * UNDERSAMPLE;
      }
      dma_bytes = samples * 4;
      break;
    default:
      dma_bytes = GSAI_DMA_BYTE_COUNT;
      break;
  }
  return dma_bytes;
}

//Util functions
static int read_from_adc(adcInfo_t* adcinfo, unsigned adc_index);

// Routine for initializing ADC modules on startup
int
iop_adc_init( adcInfo_t* adcinfo )
{
    volatile int* adcDummyData;
    int           ii, jj;
    int           dma_bytes = 32;
    int           card = 0;
    int samples;

    // Determine order to read ADC cards
    // 18bit ADC modules deliver their data 1/2 cycle later
    // than 16bit, so want to wait on their data first.
    for ( jj = 0; jj < cdsPciModules.adcCount; jj++ )
    {
        if ( cdsPciModules.adcType[ jj ] != GSC_18AI32SSC1M &&
             cdsPciModules.adcType[ jj ] != GSC_18AI64SSC )
        {
            adcinfo->adcRdOrder[ card ] = jj;
            card++;
        }
    }
    for ( jj = 0; jj < cdsPciModules.adcCount; jj++ )
    {
        if ( cdsPciModules.adcType[ jj ] == GSC_18AI32SSC1M ||
             cdsPciModules.adcType[ jj ] == GSC_18AI64SSC )
        {
            adcinfo->adcRdOrder[ card ] = jj;
            card++;
        }
    }
    /// \> If IOP,  Initialize the ADC modules
    for ( jj = 0; jj < cdsPciModules.adcCount; jj++ )
    {

        // Preload input memory with dummy variables to test that new ADC data
        // has arrived.
        // Write a dummy 0 to first ADC channel location
        // This location should never be zero when the ADC writes data as it
        // should always have an upper bit set indicating channel 0.
        // Write a number into the last channel which the ADC should never write
        // ie no upper bits should be set in channel 31.
        adcDummyData = cdsPciModules.pci_adc[ jj ];
        switch ( cdsPciModules.adcType[ jj ] )
        {
        case GSC_18AI32SSC1M:
        case GSC_18AI64SSC:
            samples = cdsPciModules.adcChannels[ jj ] * UNDERSAMPLE;
            dma_bytes = samples * 4;
            plx9056_adc_dma_setup( jj, dma_bytes );
            *adcDummyData = 0x0;
            adcDummyData += samples - 1 ;
            *adcDummyData = DUMMY_ADC_VAL;
            break;
        default:
            dma_bytes = GSAI_DMA_BYTE_COUNT;
            plx9056_adc_dma_setup( jj, dma_bytes );
            *adcDummyData = 0x0;
            adcDummyData += GSAI_64_OFFSET;
            *adcDummyData = DUMMY_ADC_VAL;
            break;
        }

        // Set ADC Present Flag
        pLocalEpics->epicsOutput.statAdc[ jj ] = ADC_MAPPED;
        // Set ADC AutoCal Pass/Fail Flag
        if ( ( cdsPciModules.adcConfig[ jj ] & GSAI_AUTO_CAL_PASS ) != 0 )
            pLocalEpics->epicsOutput.statAdc[ jj ] |= ADC_CAL_PASS;
        // Reset Diag Info
        adcinfo->adcRdTimeErr[ jj ] = 0;
        adcinfo->adcChanErr[ jj ] = 0;
        adcinfo->adcOF[ jj ] = 0;
        for ( ii = 0; ii < MAX_ADC_CHN_PER_MOD; ii++ )
            adcinfo->overflowAdc[ jj ][ ii ] = 0;
    }
    adcinfo->adcHoldTime = 0;
    adcinfo->adcHoldTimeMax = 0;
    adcinfo->adcHoldTimeEverMax = 0;
    adcinfo->adcHoldTimeEverMaxWhen = 0;
    adcinfo->adcHoldTimeMin = 0xffff;
    adcinfo->adcHoldTimeAvg = 0;
    adcinfo->adcWait = 0;
    adcinfo->chanHop = 0;

    return 0;
}

// ADC Read *****************************************************************
// Routine for reading data from ADC modules.
int
iop_adc_read( adcInfo_t* adcinfo, uint64_t cpuClock[] )
{
    int           ii, jj, kk;
    volatile int* packedData;
    int           limit;
    int           mask;
    unsigned int  offset;
    int           num_outs;
    int           ioMemCntr = 0;
    int           card;
    int           chan;
    int           max_loops = UNDERSAMPLE;  //number of samples typically processed from an A2D per cycle.
    int           loops = max_loops;     //can be less than max_loops on the first cycle when time_shift parameter is used.
    int           iocycle = 0;
    int           first_chan_marker = 0;
    static unsigned long long           adc_wait_limit = MAX_ADC_WAIT;
    double        val2dec = 0;
    static double iopdHistory[ ( MAX_ADC_MODULES * MAX_ADC_CHN_PER_MOD ) ]
                             [ MAX_HISTRY ];

    // Read ADC data
    for ( jj = 0; jj < cdsPciModules.adcCount; jj++ )
    {
        card = adcinfo->adcRdOrder[ jj ];



        /// - ---- ADC DATA RDY is detected when last channel in memory no
        /// longer contains the dummy variable written during initialization and
        /// reset after each read.
        packedData = cdsPciModules.pci_adc[ card ];

        /// Where last channel is is dependent on adcDummyDatanumber of channels a
        /// particular card type has.
        switch ( cdsPciModules.adcType[ card ] )
        {
        case GSC_18AI32SSC1M:
            if( (adc_read_state == ADC_READ_ADJUST_TIME_SHIFT) && cdsPciModules.adcTimeShift[card] == UNDERSAMPLE )
            {
                // we'll re-use the first DMA result next IOP cycle.
                continue;
            }
            if(adc_read_state == ADC_READ_ADJUST_TIME_SHIFT)
            {
                packedData +=
                    ( cdsPciModules.adcChannels[ card ] * (UNDERSAMPLE
                                - cdsPciModules.adcTimeShift[card]) - 1 );
            }
            else
            {
                packedData +=
                    ( cdsPciModules.adcChannels[ card ] * UNDERSAMPLE - 1 );
            }
            first_chan_marker = GSAF_FIRST_SAMPLE_MARK;
            break;
        case GSC_18AI64SSC:
            if( (adc_read_state == ADC_READ_ADJUST_TIME_SHIFT) && cdsPciModules.adcTimeShift[card] == UNDERSAMPLE )
            {
                // we'll re-use the first DMA result next IOP cycle.
                continue;
            }
            if(adc_read_state == ADC_READ_ADJUST_TIME_SHIFT)
            {
                packedData +=
                    ( cdsPciModules.adcChannels[ card ] *( UNDERSAMPLE
                      - cdsPciModules.adcTimeShift[card] )- 1 );
            }
            else
            {
                packedData +=
                    ( cdsPciModules.adcChannels[ card ] * UNDERSAMPLE - 1 );
            }
            first_chan_marker = GSA7_FIRST_SAMPLE_MARK;
            break;
        default:
            if((adc_read_state == ADC_READ_ADJUST_TIME_SHIFT) && cdsPciModules.adcTimeShift[card] == 1)
            {
                continue;
            }
            packedData += GSAI_64_OFFSET;
            first_chan_marker = GSAI_FIRST_SAMPLE_MARK;
            break;
        }

        // Capture CPU clock cycle for timing diags
        timer_start( &cpuClock[ CPU_TIME_RDY_ADC ] );
        // Wait for ADC data to arrive via DMA
        do
        {
            /// - ---- Need to delay if not ready as constant banging of the
            /// input register will slow down the ADC DMA.
            timer_start( &cpuClock[ CPU_TIME_ADC_WAIT ] );
            adcinfo->adcWait = timer_tock_ns( &cpuClock[ CPU_TIME_RDY_ADC ] ) / 1000;
        } while ( ( *packedData == DUMMY_ADC_VAL ) &&
                  ( adcinfo->adcWait < adc_wait_limit ) );


        /// - ---- Added ADC timing diagnostics to verify timing consistent and
        /// all rdy together.
        if ( jj == 0 )
        {
            //Calculate the time (in us) between last cycles start time and when 
            //the ADC data was ready for this cycle
            adcinfo->adcRdTime[ card ] = timer_duration_ns( &cpuClock[ CPU_TIME_CYCLE_START ], 
                                                            &cpuClock[ CPU_TIME_ADC_WAIT ]) / 1000;


#ifdef XMIT_DOLPHIN_TIME
            // Send time on Dolphin net if this is the time xmitter.
            if ( pLocalEpics->epicsOutput.disableRemoteIpc == 0) {
                pcieTimer->gps_time = timeSec;
                pcieTimer->cycle = cycleNum/UNDERSAMPLE;
                clflush_cache_range( (void*)&pcieTimer->gps_time, 16 );
            }
#endif
        }
        else
        {
            adcinfo->adcRdTime[ card ] = adcinfo->adcWait;
        }

        if ( adcinfo->adcRdTime[ card ] > adcinfo->adcRdTimeMax[ card ] )
            adcinfo->adcRdTimeMax[ card ] = adcinfo->adcRdTime[ card ];

        if ( ( jj == 0 ) &&
            ( adcinfo->adcRdTimeMax[ card ] >
            ( MAX_ADC_WAIT_CARD_0 * UNDERSAMPLE ) ) )
                adcinfo->adcRdTimeErr[ card ]++;

        if ( ( jj != 0 ) &&
            ( adcinfo->adcRdTimeMax[ card ] > MAX_ADC_WAIT_CARD_S ) )
                adcinfo->adcRdTimeErr[ card ]++;

        /// - --------- If data not ready in time, abort.
        /// Either the clock is missing or code is running too slow and ADC FIFO
        /// is overflowing.
        if ( adcinfo->adcWait >= adc_wait_limit )
        {
            pLocalEpics->epicsOutput.statAdc[ card ] &= ~( ADC_RD_TIME );
            fe_status_return = ADC_TO_ERROR;
            fe_status_return_subcode = card;
            return ( ADC_TO_ERROR );
        }


        if ( jj == 0 ) // Capture timing info
        {
            // Capture start of cycle time for cpu meter diagnostics
            timer_start( &cpuClock[ CPU_TIME_CYCLE_START ] );
            /// \> If first cycle of a new second, capture time from IRIG-B
            /// module or LIGO PCIe timing receiver.
            if ( cycleNum == 0 )
            {
                // if SymCom type, just do write to lock current time and read
                // later This save a couple three microseconds here
                switch ( cdsPciModules.gpsType )
                {
                case SYMCOM_RCVR:
                    lockGpsTime( &cdsPciModules );
                    break;
                case TSYNC_RCVR:
                    /// - ---- Reading second info will lock the time register,
                    /// allowing nanoseconds to be read later (on next cycle).
                    /// Two step process used to save CPU time here, as each
                    /// read can take 2usec or more.
                    timeSec = getGpsSecTsync( &cdsPciModules );
                    break;
                case LIGO_RCVR:
                    // Read GPS second from LIGO Pcie timing card.
                    timeSec = lptc_get_gps_sec( &cdsPciModules );
                    break;
                default:
                    break;
                }
            }
        }

        /// \> Read adc data
        // Set pointer to first channel
        packedData = cdsPciModules.pci_adc[ card ];
        /// - ---- First, and only first, channel should have upper bit marker
        /// set. If not, have a channel hopping error.
        if ( !( *packedData & first_chan_marker ) )
        {
            adcinfo->chanHop = 1;
            adcinfo->adcChanErr[ card ] = 1;
            fe_status_return = CHAN_HOP_ERROR;
            fe_status_return_subcode = card;
            return CHAN_HOP_ERROR;
        }

        // Various ADC models have different number of channels/data bits
        if ( cdsPciModules.adcType[ card ] == GSC_16AI64SSA )
        {
            limit = OVERFLOW_LIMIT_16BIT;
            offset = GSAI_DATA_CODE_OFFSET;
            mask = GSAI_DATA_MASK;
        }
        else
        {
            //Both the GSC_18AI64SSC and GSC_18AI32SSC1M have the same bit resolution
            limit = OVERFLOW_LIMIT_18BIT;
            offset = GSA7_DATA_CODE_OFFSET;
            mask = GSA7_DATA_MASK;
        }
        num_outs = cdsPciModules.adcChannels[ card ];

        // loops and max_loops are usually equal to UNDERSAMPLE, but on the first cycle
        // they loops may be less.
        max_loops = UNDERSAMPLE;
        if(adc_read_state == ADC_READ_ADJUST_TIME_SHIFT)
        {
            loops = UNDERSAMPLE - cdsPciModules.adcTimeShift[card];
        }
        else
        {
            loops = UNDERSAMPLE;
        }

        // GSC_16AI64SSA can't support rate > 128KS/sec
        // Even if IOP rate is faster, read these ADCs at 65kHz.
        if ( cdsPciModules.adcType[ card ] == GSC_16AI64SSA && UNDERSAMPLE > 4 )
        {
            max_loops = 1;
            loops = 1;
        }

        /// - ---- Determine next ipc memory location to load ADC data
        ioMemCntr = ( (int)( cycleNum / ADC_MEMCPY_RATE ) % IO_MEMORY_SLOTS );
        iocycle = (int)( cycleNum / ADC_MEMCPY_RATE );


        /// - ----  Read adc data from PCI mapped memory into local variables
        for ( kk = 0; kk < loops; kk++ )
        {
            for ( chan = 0; chan < num_outs; chan++ )
            {
                // adcData is the integer representation of the ADC data
                adcinfo->adcData[ card ][ chan ] = ( *packedData & mask );
                adcinfo->adcData[ card ][ chan ] -= offset;
#ifdef DEC_TEST
                // This only used on test system for checking decimation filters
                // by providing an ADC signal via a GDS EXC signal
                if ( chan == 0 )
                {
                    adcinfo->adcData[ card ][ chan ] =
                        dspPtr[ 0 ]->data[ 0 ].exciteInput;
                }
#endif
                // dWord is the double representation of the ADC data
                // This is the value used by the rest of the code calculations.
                dWord[ card ][ chan ][ kk ] = adcinfo->adcData[ card ][ chan ];

                // If running with fast ADC at 512K and gsc16ai64 at 64K, then:
                // fill in the missing data by doing data copy
                if ( cdsPciModules.adcType[ card ] == GSC_16AI64SSA &&
                     UNDERSAMPLE > 4 )
                {
                    for ( ii = 1; ii < UNDERSAMPLE; ii++ )
                        dWord[ card ][ chan ][ ii ] =
                            adcinfo->adcData[ card ][ chan ];
                }
                /// - ----  Load ADC value into ipc shared memory buffer
                if ( max_loops == 8 )
                {
                    val2dec = adcinfo->adcData[ card ][ chan ];
                    ioMemData->inputData[ card ][ ioMemCntr ].data[ chan ] =
                        (int)iir_filter_biquad(
                            val2dec,
                            feCoeff8x,
                            3,
                            &iopdHistory[ chan + card * MAX_ADC_CHN_PER_MOD ]
                                        [ 0 ] );
                }
                else
                {
                    ioMemData->inputData[ card ][ ioMemCntr ].data[ chan ] =
                        adcinfo->adcData[ card ][ chan ];
                }

                /// - ---- Check for ADC overflows
                if ( ( adcinfo->adcData[ card ][ chan ] > limit ) ||
                     ( adcinfo->adcData[ card ][ chan ] < -limit ) )
                {
                    adcinfo->overflowAdc[ card ][ chan ]++;
                    pLocalEpics->epicsOutput.overflowAdcAcc[ card ][ chan ]++;
                    overflowAcc++;
                    adcinfo->adcOF[ card ] = 1;
                    odcStateWord |= ODC_ADC_OVF;
                }

                packedData++;




            } //End of loop for each chan

            // For normal IOP ADC undersampling ie not a mixed fast 512K adc and
            // normal 64K adc
            if ( UNDERSAMPLE < 5 )
            {
                /// - ---- Write GPS time and cycle count as indicator to
                /// control app that adc data is ready
                ioMemData->gpsSecond = timeSec;
                ioMemData->inputData[ card ][ ioMemCntr ].timeSec = timeSec;
                ioMemData->inputData[ card ][ ioMemCntr ].cycle = iocycle;
                ioMemCntr++;
                iocycle++;
                iocycle %= IOP_IO_RATE;
            }
        }

        // For ADC undersampling when running with a fast ADC at 512K
        // to limit rate to user apps at 64K
        if ( UNDERSAMPLE > 4 )
        {
            /// - ---- Write GPS time and cycle count as indicator to control
            /// app that adc data is ready
            ioMemData->gpsSecond = timeSec;
            ioMemData->inputData[ card ][ ioMemCntr ].timeSec = timeSec;
            ioMemData->inputData[ card ][ ioMemCntr ].cycle = iocycle;
        }

        /// - ---- Clear out 1st channel from last ADC data read for test on
        /// next cycle
        /// - ---- First channel should never be zero ie should have channel
        /// marker bit
        packedData = cdsPciModules.pci_adc[ card ];
        *packedData = 0x0;


        // Enable the ADC Demand DMA for next read
        switch ( cdsPciModules.adcType[ card ] )
        {
        case GSC_18AI32SSC1M:
        case GSC_18AI64SSC:
            packedData +=
                ( cdsPciModules.adcChannels[ card ] * UNDERSAMPLE - 1 );
            *packedData = DUMMY_ADC_VAL;
            if ( adc_read_state == ADC_READ_PREP_TIME_SHIFT ) {
                plx9056_adc_dma_set_size(card,
                                         iop_adc_timeshift_dma_size(card) );
            }
            else if ( adc_read_state == ADC_READ_ADJUST_TIME_SHIFT )
            {
                plx9056_adc_dma_set_size(card,
                          cdsPciModules.adcChannels[ card ] * UNDERSAMPLE * 4 );
            }
            plx9056_adc_dma_start( card );
            break;
        default:
            packedData += GSAI_64_OFFSET;
            *packedData = DUMMY_ADC_VAL;
            plx9056_adc_dma_start( card );
            break;
        }

#ifdef DIAG_TEST
        // For DIAGS ONLY !!!!!!!!
        // This will change ADC DMA BYTE count
        // -- Greater than normal will result in channel hopping.
        // -- Less than normal will result in ADC timeout.
        // In both cases, real-time kernel code should exit with errors to dmesg
        if ( pLocalEpics->epicsInput.bumpAdcRd != 0 )
        {
            gsc16ai64DmaBump( card, pLocalEpics->epicsInput.bumpAdcRd );
            pLocalEpics->epicsInput.bumpAdcRd = 0;
        }
#endif
        adc_wait_limit = MAX_ADC_WAIT_CARD_0;
    }

    // progress in order from ADC_READ_PREP_TIME_SHIFT -> ADC_READ_ADJUST_TIME_SHIFT -> ADC_READ_RUNNING
    // then stay at ADC_READ_RUNNING
    if (adc_read_state > ADC_READ_RUNNING) {
      adc_read_state -= 1;
      RTSLOG_DEBUG("adc_read_state = %d", adc_read_state);
    }
    return 0;
}


int iop_adc_sync_cards( adcInfo_t* adcinfo ) 
{

    //We need to read from ADC 0 and wait the expected time.
    //If we have a fast read, we might just need to clear buffers

    int card, jj, not_synced=1, attempts=0;
    int slow_cards[MAX_ADC_MODULES] = {0}; //ADC 0 should be slow
    volatile int* packedData;
    int first_chan_marker = 0;


    while( not_synced && attempts < 10 ) {
        ++attempts;

        for ( jj = 0; jj < cdsPciModules.adcCount; jj++ )
        {
            card = adcinfo->adcRdOrder[ jj ];

            packedData = cdsPciModules.pci_adc[ card ];
            
            /// Where last channel is is dependent on adcDummyDatanumber of channels a
            /// particular card type has.
            switch ( cdsPciModules.adcType[ card ] )
            {
                case GSC_18AI32SSC1M:
                    packedData +=
                        ( cdsPciModules.adcChannels[ card ] * UNDERSAMPLE - 1 );
                    first_chan_marker = GSAF_FIRST_SAMPLE_MARK;
                    break;
                case GSC_18AI64SSC:
                    packedData +=
                        ( cdsPciModules.adcChannels[ card ] * UNDERSAMPLE - 1 );
                    first_chan_marker = GSA7_FIRST_SAMPLE_MARK;
                    break;
                default:
                    packedData += GSAI_64_OFFSET;
                    first_chan_marker = GSAI_FIRST_SAMPLE_MARK;
                    break;
            }

            uint64_t wait_start_tsc;
            unsigned adc_wait_time_ns=0, adc_wait_limit_ns = MAX_ADC_WAIT_CARD_S * 1000;
            if( jj == 0) {
                adc_wait_limit_ns = MAX_ADC_WAIT_CARD_0*1000;
            }

            // Capture CPU clock cycle for timing diags
            timer_start( &wait_start_tsc);
            // Wait for ADC data to arrive via DMA
            while ( ( *packedData == DUMMY_ADC_VAL ) &&
                    ( adc_wait_time_ns < adc_wait_limit_ns ) );
            {
                udelay(1);
                adc_wait_time_ns = timer_tock_ns( &wait_start_tsc );
            }
            RTSLOG_DEBUG("iop_adc_sync_cards() - It took %d ns to read card %d\n", adc_wait_time_ns, card);
            if( adc_wait_time_ns >= adc_wait_limit_ns) {
                slow_cards[jj] = 1;
            }


            packedData = cdsPciModules.pci_adc[ card ];
            *packedData = 0x0;


            // Enable the ADC Demand DMA for next read
            switch ( cdsPciModules.adcType[ card ] )
            {
            case GSC_18AI32SSC1M:
            case GSC_18AI64SSC:
                packedData +=
                    ( cdsPciModules.adcChannels[ card ] * UNDERSAMPLE - 1 );
                *packedData = DUMMY_ADC_VAL;
                plx9056_adc_dma_start( card );
                break;
            default:
                packedData += GSAI_64_OFFSET;
                *packedData = DUMMY_ADC_VAL;
                plx9056_adc_dma_start( card );
                break;
            }

        } //End loop over ADCs

        //If we had to wait longer for any other card than the first one,
        //than means we are unsynchronized
        not_synced = 0;
        for(jj=1; jj<cdsPciModules.adcCount; ++jj) {
            if ( slow_cards[jj] == 1) {
                not_synced = 1;
                RTSLOG_INFO("iop_adc_sync_cards() - Unsynchronized card detected at read order %d, card %d \n", jj, card);
                
            }
        }
        if ( not_synced ) { //Read from all fast ADCs
            for(jj=0; jj<cdsPciModules.adcCount; ++jj) {
                if( slow_cards[jj] == 0 ) 
                 read_from_adc(adcinfo, jj);

                slow_cards[jj] = 0;//Reset for next loop
            }
        }


    } //While not synced


    if( attempts >= 10 ) return -1;

    RTSLOG_INFO("iop_adc_sync_cards() - Cards now appear to be synchronized.\n");
    return 0;
}

static int read_from_adc(adcInfo_t* adcinfo, unsigned adc_index) 
{
    volatile int* packedData;
    int card = adcinfo->adcRdOrder[ adc_index ];
    int first_chan_marker;

    packedData = cdsPciModules.pci_adc[ card ];
    
    /// Where last channel is is dependent on adcDummyDatanumber of channels a
    /// particular card type has.
    switch ( cdsPciModules.adcType[ card ] )
    {
        case GSC_18AI32SSC1M:

            packedData +=
                ( cdsPciModules.adcChannels[ card ] * UNDERSAMPLE - 1 );
        
            first_chan_marker = GSAF_FIRST_SAMPLE_MARK;
            break;
        case GSC_18AI64SSC:

            packedData +=
                ( cdsPciModules.adcChannels[ card ] * UNDERSAMPLE - 1 );

            first_chan_marker = GSA7_FIRST_SAMPLE_MARK;
            break;
        default:

            packedData += GSAI_64_OFFSET;
            first_chan_marker = GSAI_FIRST_SAMPLE_MARK;
            break;
    }

    uint64_t wait_start_tsc;
    uint64_t adc_wait_time_ns=0, adc_wait_limit_ns = USEC_PER_CYCLE + 10;


    // Capture CPU clock cycle for timing diags
    timer_start( &wait_start_tsc);
    // Wait for ADC data to arrive via DMA
    while ( ( *packedData == DUMMY_ADC_VAL ) &&
            ( adc_wait_time_ns < adc_wait_limit_ns ) );
    {
        udelay(1);
        adc_wait_time_ns = timer_tock_ns( &wait_start_tsc );
    }
    RTSLOG_DEBUG("It took %lu ns to read card %d in single ADC read\n", adc_wait_time_ns, card);


    packedData = cdsPciModules.pci_adc[ card ];
    *packedData = 0x0;


    // Enable the ADC Demand DMA for next read
    switch ( cdsPciModules.adcType[ card ] )
    {
    case GSC_18AI32SSC1M:
    case GSC_18AI64SSC:
        packedData +=
            ( cdsPciModules.adcChannels[ card ] * UNDERSAMPLE - 1 );
        *packedData = DUMMY_ADC_VAL;
        plx9056_adc_dma_start( card );
        break;
    default:
        packedData += GSAI_64_OFFSET;
        *packedData = DUMMY_ADC_VAL;
        plx9056_adc_dma_start( card );
        break;
    }

    return 0;
}