#ifndef RAII_HH
#define RAII_HH

#include <unistd.h>
#include <memory>

#include "util/user/raii.hh"

/**
 * Some 'smart' resource wrappers using the raii technique.
 * This should be cleaned up and folded into C++11 code as
 * it becomes feasible.
 */

namespace raii
{
    template < typename T >
    class c_unique_ptr
    {
    public:
        using pointer = T*;
        using element_type = T;

        c_unique_ptr( ) noexcept : p_{ nullptr }
        {
        }

        template < typename... Ts >
        static c_unique_ptr
        make( Ts&&... params )
        {
            c_unique_ptr p{ };
            auto         mem = reinterpret_cast< T* >( malloc( sizeof( T ) ) );
            if ( !mem )
            {
                throw std::runtime_error( "unable to allocate object" );
            }
            try
            {
                new ( mem ) T( std::forward< Ts >( params )... );
            }
            catch ( ... )
            {
                free( mem );
                throw;
            }
            p.p_ = mem;
            mem = nullptr;
            return p;
        }

        c_unique_ptr( c_unique_ptr&& other ) noexcept : p_{ other.p_ }
        {
            other.p_ = nullptr;
        }
        c_unique_ptr( const c_unique_ptr& ) = delete;
        ~c_unique_ptr( )
        {
            if ( p_ )
            {
                p_->~T( );
                free( p_ );
                p_ = nullptr;
            }
        }

        c_unique_ptr& operator=( const c_unique_ptr& ) = delete;
        c_unique_ptr&
        operator=( c_unique_ptr&& other ) noexcept
        {
            if ( this != &other )
            {
                reset( );
                p_ = other.p_;
                other.p_ = nullptr;
            }
            return *this;
        }

        void
        swap( c_unique_ptr& other ) noexcept
        {
            std::swap( other.p_, p_ );
        }

        void
        reset( ) noexcept
        {
            if ( p_ )
            {
                extract_ptr( );
            }
        }

        pointer
        release( ) noexcept
        {
            pointer p = p_;
            p_ = nullptr;
            return p;
        }

        pointer
        get( ) const noexcept
        {
            return p_;
        }

        T&
        operator*( ) const noexcept
        {
            return *p_;
        }

        pointer
        operator->( ) const noexcept
        {
            return p_;
        }

        explicit operator bool( ) const noexcept
        {
            return p_ != nullptr;
        }

    private:
        c_unique_ptr
        extract_ptr( ) noexcept
        {
            c_unique_ptr ptr{ };
            ptr.p_ = p_;
            p_ = nullptr;
            return ptr;
        }

        T* p_{ nullptr };
    };

} // namespace raii

#endif // RAII_HH
