#ifndef LIGO_PCIE_TIMING_CARD_HEADER_H
#define LIGO_PCIE_TIMING_CARD_HEADER_H
// LIGO PCIe Timing Card
//

// PCI identifiers
#define LPTC_VID    0x10ee
#define LPTC_TID    0xd8c6

// LPTC board id in firmware version
// can be read as lptc_get_sowftware_id() >> 4
// if not equal to this value, you don't have a well programmed FPGA
#define LPTC_BOARD_ID 0x2000337

#include "drv/cdsHardware.h"

#include <linux/types.h>
#include <linux/pci.h>

#ifdef __cplusplus
extern "C" {
#endif

// Prototypes for routines in ligoPcieTiming_core.c
int lptcInit( CDS_HARDWARE* , struct pci_dev* );
int lptc_get_gps_time( CDS_HARDWARE* , u32* , u32* );
unsigned int lptc_get_gps_usec( CDS_HARDWARE* );
unsigned int lptc_get_gps_sec( CDS_HARDWARE* );
int lptc_get_lptc_status( CDS_HARDWARE* );
int lptc_get_bp_status( CDS_HARDWARE* );
int lptc_get_bp_config( CDS_HARDWARE* );
void lptc_set_bp_config( CDS_HARDWARE* pCDS, u32 value);
void lptc_set_wd_reset( CDS_HARDWARE* pCds, u32 value);
int lptc_get_slot_phase( CDS_HARDWARE* pCds, int slot );
void lptc_set_slot_phase( CDS_HARDWARE* pCds, int slot, u32 value);


// slot functions in ligoPcieTiming_core.c
int lptc_get_slot_config( CDS_HARDWARE* , int );
void lptc_set_slot_config( CDS_HARDWARE* pCds, int slot, u32 value );
int lptc_get_slot_status( CDS_HARDWARE* , int );

// advanced timing functions in ligoPcieTiming_core.c
int lptc_get_adv_timing_config( CDS_HARDWARE* );
void lptc_set_adv_timing_config( CDS_HARDWARE*, u32 value );
int lptc_get_node_address( CDS_HARDWARE* );
void lptc_set_node_address( CDS_HARDWARE*, u32 value );
int lptc_get_adv_timing_status( CDS_HARDWARE* );
int lptc_get_duotone_shift( CDS_HARDWARE* );
void lptc_set_duotone_shift( CDS_HARDWARE*, u32 value);
int lptc_get_adv_board_id( CDS_HARDWARE* );
int lptc_get_adv_software_id( CDS_HARDWARE* );
int lptc_get_vcxo_control_volts( CDS_HARDWARE* );

// diagnostics functions in ligoPcieTiming_core.c
int lptc_get_board_id( CDS_HARDWARE* pCds );
int lptc_get_board_sn( CDS_HARDWARE* pCds );
int lptc_get_software_id( CDS_HARDWARE* pCds );
int lptc_get_software_rev( CDS_HARDWARE* pCds );
int lptc_get_gps_sec_diag( CDS_HARDWARE* pCds );
int lptc_get_mod_address( CDS_HARDWARE* pCds );
int lptc_get_board_status( CDS_HARDWARE* pCds );
int lptc_get_board_config( CDS_HARDWARE* pCds );
int lptc_get_ocxo_controls( CDS_HARDWARE* pCds );
int lptc_get_ocxo_error( CDS_HARDWARE* pCds );
int lptc_get_uplink_1pps_delay( CDS_HARDWARE* pCds );
int lptc_get_external_1pps_delay( CDS_HARDWARE* pCds );
int lptc_get_gps_1pps_delay( CDS_HARDWARE* pCds );
int lptc_get_fanout_up_loss( CDS_HARDWARE* pCds );
int lptc_get_fanout_missing_delay_error( CDS_HARDWARE* pCds );
int lptc_get_leaps_and_error( CDS_HARDWARE* pCds );



// 'On Board Features' (LIGO-T2000406 3.9)
// in ligoPcieTimeing_core.c
int lptc_get_brd_synch_factors( CDS_HARDWARE* pCds );
void lptc_set_brd_synch_factors( CDS_HARDWARE* pCds, u32 value);
int lptc_get_xadc_config( CDS_HARDWARE* pCds );
void lptc_set_xadc_config( CDS_HARDWARE* pCds, u32 value);
int lptc_get_board_and_powersupply_status( CDS_HARDWARE* pCds );
int lptc_get_xadc_status( CDS_HARDWARE* pCds );
//temperature from 0x0190 offset
int lptc_get_temp( CDS_HARDWARE* pCds );
//internal power values from 0x0190 and 0x0194.
void lptc_get_internal_pwr( CDS_HARDWARE* pCds, u32 *output );
// ouput should be a preallocated array the same size as
// LPTC_OBF_REGISTER:EPS
void lptc_get_external_pwr( CDS_HARDWARE* pCds, u32 *output);

// duotone control
int lptc_can_configure_duotone(CDS_HARDWARE* pCds);
void lptc_set_duotone_amplitude(CDS_HARDWARE* pCds, u32 value);
int lptc_get_duotone_amplitude(CDS_HARDWARE* pCds);
void lptc_set_duotone_frequency(CDS_HARDWARE* pCds, u32 value);
int lptc_get_duotone_frequency(CDS_HARDWARE* pCds);


// Prototypes for routine in ligoPcieTiming.c
void lptc_enable_all_slots( CDS_HARDWARE* );
int lptc_start_clock( CDS_HARDWARE* );
void lptc_status_update( CDS_HARDWARE* );
int lptc_stop_clock( CDS_HARDWARE* );
void lptc_dac_duotone( CDS_HARDWARE* , int );
void lptc_slot_clk_set( CDS_HARDWARE* , int, int );
void lptc_slot_clk_disable_all( CDS_HARDWARE* );
void lptc_slot_clk_enable_all( CDS_HARDWARE* );
int lptc_wait_for_dac_sync( CDS_HARDWARE* );
int lptc_get_temp_c( CDS_HARDWARE* pCds );

/**
* @brief Queries the timing card for duotone parameters that should
*        be used for its version of firmware.
*
* @param pCds [in] A pointer to the models CDS_HARDWARE structure
*
* @param dt_offset_usec_ptr [out] The offset (in us) that should be used
* @param dt_data_count_ptr [out] The number of data points that should be used 
*
* @return 0 if the firmware version is know and supported, -1 otherwise
*/
int lptc_get_duotone_info( CDS_HARDWARE* pCds,
                           double * dt_offset_usec_ptr,
                           int * dt_data_count_ptr,
                           int * adc_offset_ptr
                           );

#ifdef __cplusplus
}
#endif




#define LPTC_BP_SLOTS 10
#define LPTC_OBF_EPS_REGS 8


typedef struct SLOT_CONFIG {
    u32 config;
    u32 phase;
    u32 status;
    u32 reserved;
}SLOT_CONFIG;


typedef struct LPTC_REGISTER {
    u64 gps_time;        // 0x0000
    u32 status;          // 0x0008
    u32 revision;        // 0x000c
    u32 bp_config;       // 0x0010
    u32 wd_reset;        // 0x0014
    u32 bp_status;       // 0x0018
    u32 reserved3;       // 0x001c
    SLOT_CONFIG slot_info[LPTC_BP_SLOTS];    // 0x0020
}LPTC_REGISTER;

typedef struct MSI_CONFIG {
    u32 msi_control;
    u32 msi_phase_shift;
    u32 msi_status;
    u32 reserved;
} MSI_CONFIG;

typedef struct LPTC_INTERRUPT {
    MSI_CONFIG msi_config[4]  ;   // 0x00c0
} LPTC_INTERRUPT;

typedef struct LPTC_GPS_EXPANSION {
    u32 gps_config;       // 0x0100
    u32 preset_frequency;  // 0x0104
    u32 gps_status;       // 0x0108
    u32 freq_counter;     // 0x010C
} LPTC_GPS_EXPANSION;

typedef struct LPTC_FANOUT_EXPANSION {
    u32 fanout_config;    // 0x0110
    u32 irigb_config;     // 0x0114
    u32 fanout_status;    // 0x0118
    u32 int_ext_delay;    // 0x011c
    u32 int_int_delay;    // 0x0120
    u32 decoded_irigb[3];  // 0x0124
} LPTC_FANOUT_EXPANSION;

typedef struct LPTC_ADVANCED_TIMING {
    u32 adv_tim_config;   // 0x130
    u32 node_address;     // 0x134
    u32 adv_tim_status;   // 0x138
    u32 duotone_shift;    // 0x13c
    u32 board_id_rev;     // 0x140
    u32 software_id_rev;  // 0x144
    u32 vcxo_ctrl_volt;   // 0x148
    u32 duotone_config;   // 0x14c
} LPTC_ADVANCED_TIMING;

typedef struct LPTC_OBF_REGISTER {
    u32 brd_config;    // 0x0180
    u32 xadc_config;   // 0x0184
    u32 bps_status;    // 0x0188
    u32 xadc_status;   // 0x018c
    u32 tips1;         // 0x0190
    u32 ips2;          // 0x0194
    u32 eps[LPTC_OBF_EPS_REGS];        // 0x0198
} LPTC_OBF_REGISTER;

// Data read from LPTC at 0x1000 offset
typedef struct LPTC_DIAG_INFO {
    u32 board_id;        // 0x1000
    u32 board_sn;        // 0x1004
    u32 sftware_id;      // 0x1008
    u32 sftware_rev;     // 0x100c
    u32 gps_seconds;     // 0x1010
    u32 mod_address;     // 0x1014
    u32 board_status;    // 0x1018
    u32 board_config;    // 0x101c
    u32 ocxo_controls;        // 0x1020
    u32 ocxo_error;           // 0x1024
    u32 uplink_1pps_delay;    // 0x1028
    u32 external_1pps_delay;  //0x102c
    u32 gps_1pps_delay;       //0x1030
    u32 fanout_up_loss;       //0x1034
    u32 fanout_missing_delay_error; //0x1038
    u32 leaps_and_error;      //0x103c
}LPTC_DIAG_INFO;

//complete lptc register space
typedef struct LPTC_REGISTER_SPACE {
    LPTC_REGISTER lptc_registers;   //ends at 0xc0
    LPTC_INTERRUPT interrupt_config;  //starts at 0xc0, ends at 0x100
    LPTC_GPS_EXPANSION gps_expansion; //starts at 0x100, ends at 0x110
    LPTC_FANOUT_EXPANSION fanout_expansion; //starts at 0x110, ends at 0x0130
    struct LPTC_ADVANCED_TIMING advanced_timing; //starts at 0x0130, ends at 0x0150
    u64 unused[0x6];   //another 0x30 bytes gets to offset 0x180
    LPTC_OBF_REGISTER obf_registers; //starts at 0x180, ends at 0x1a4
    u32 unused2[0x392];  //another 0xe48 bytes gets to 0x1000.
    LPTC_DIAG_INFO diag_info;
} LPTC_REGISTER_SPACE;

// Slot configuration register bits
// Bit 0-7 = Frequency 
#define LPTC_SCR_CLK_ENABLE         0x100       // Bit 8 : Enable clock signal
#define LPTC_SCR_CLK_INVERT         0x200       // Bit 9 : Inverted clock signal
#define LPTC_SCR_CLK_1PPS           0x400       // Bit 10: Start clock at next second boundary
#define LPTC_SCR_IDLE_TRANS         0x800       // Bit 11: Start clock at next transition from idle
#define LPTC_SCR_IDLE_HIGH          0x1000      // Bit 12: Full Idle clock line high
#define LPTC_SCR_TIM_SIG            0x2000      // Bit 13: Use timing signal as clock
//                                                 Bit 14-15: Reserved
#define LPTC_SCR_LVDS               0x10000     // Bit 16: Use LVDS clock lines
#define LPTC_SCR_ADC_DT_ENABLE      0x20000     // Bit 17: Enable ADC Duotone on last ADC chan
#define LPTC_SCR_DAC_DT_ENABLE      0x40000     // Bit 18: Enable DAC Duotone on last -1  ADC chan

#define LPTC_SCR_ADC_SET            (LPTC_SCR_CLK_ENABLE | LPTC_SCR_CLK_INVERT | LPTC_SCR_IDLE_HIGH)
#define LPTC_SCR_DAC_SET            (LPTC_SCR_CLK_ENABLE | LPTC_SCR_IDLE_HIGH)

// LPTC Status register at 0x0008
#define LPTC_STATUS_OK                0x80000000  // Bit 31
#define LPTC_STATUS_ROOT_NODE         0x40000000  // Bit 30
#define LPTC_STATUS_SUP_FANOUT        0x20000000  // Bit 29
#define LPTC_STATUS_UPLINK_OK         0x10000000  // Bit 28
#define LPTC_STATUS_UPLINK_LOSS       0x8000000   // Bit 27
#define LPTC_STATUS_OCXO_LOCK         0x4000000   // Bit 26
#define LPTC_STATUS_GPS_LOCK          0x2000000   // Bit 25
#define LPTC_STATUS_VCXO_VOOR         0x1000000   // Bit 24
#define LPTC_STATUS_UTC_TIME          0x800000    // Bit 23
#define LPTC_STATUS_LSEC_DECODE       0x400000    // Bit 22
#define LPTC_STATUS_LSEC_SUB_PEND     0x200000    // Bit 21
#define LPTC_STATUS_LSEC_ADD_PEND     0x100000    // Bit 20
#define LPTC_STATUS_LEAP_SEC          0xff00

// LPTC Backplane status register at 0x0018
#define LPTC_BPS_BP_PRESENT        0x200
#define LPTC_BPS_BP_REV            0x18
#define LPTC_BPS_CLK_ACTIVE        0x1
#define LPTC_BPS_CLK_RUN           0x2
#define LPTC_BPS_WD_MON            0x4

#define LPTC_CMD_STOP_CLK_ALL            0x0
#define LPTC_CMD_START_ALL_ON_NEXT_SEC   0x8
#define LPTC_CMD_ENABLE_ALL_SLOTS        0x4
#define LPTC_CMD_START_CLK_ALL             ( LPTC_CMD_START_ALL_ON_NEXT_SEC | LPTC_CMD_ENABLE_ALL_SLOTS)

#define LPTC_ENABLE_DAC_DUO        0x4000
// Slot Clock Speeds
#define LPTC_SCR_CLK_64K    0x10

#define LIGO_RCVR             0x3
#define LPTC_IOC_SLOTS         10
#define LPTC_SLOT_ENABLE        1
#define LPTC_SLOT_DISABLE       0

// Duotone frequency settings
#define LPTC_DT_FREQ_960        0
#define LPTC_DT_FREQ_1920       1
#define LPTC_DT_FREQ_3840       2
#define LPTC_DT_FREQ_15424      3

#endif /* LIGO_PCIE_TIMING_CARD_HEADER_H */
