//
// Created by jonathan.hanks on 12/1/25.
//
//
// Created by jonathan.hanks on 2/3/22.
//
//
// Created by jonathan.hanks on 2/3/22.
//
#include "catch.hpp"
#include "frame_utils.hh"
#include <algorithm>
#include <atomic>
#include <chrono>
#include <iterator>
#include <thread>
#include <string>
#include <vector>
#include <boost/filesystem.hpp>
#include <boost/process.hpp>
#include <boost/process/child.hpp>
#include <boost/process/start_dir.hpp>

#include "test_helpers.hh"
#include "simple_http.hh"
#include <daqd_stream/daqd_stream_server.hh>

namespace bp=boost::process;

std::vector< std::string >
pick_channels( const std::vector< daqd_stream::online_channel >& channels )
{
    std::vector< std::string > picks{ };
    for ( auto i = 0; i < 5; ++i )
    {
        picks.emplace_back( channels.at( i * 5 ).name.data( ) );
    }
    return picks;
}

bp::child
start_process( const std::string& process_name, const std::vector<std::string>& args, const std::string& cwd, const std::string& log_dir)
{
    std::vector<std::string> cmd{};

    std::copy(args.begin(), args.end(), std::back_inserter(cmd));
    std::cerr << "Starting process ";
    std::copy(cmd.begin( ), cmd.end( ), std::ostream_iterator<std::string>(std::cerr, " " ) );
    std::cerr << std::endl;

    auto target = bp::search_path(process_name);

    // reminder on how to set outputs and directories
    // https://stackoverflow.com/questions/30112898/boost-process-how-do-i-redirect-process-output-to-a-file
    // https://stackoverflow.com/questions/70029939/set-the-working-directory-when-starting-a-process-with-boost
    auto out_file = log_dir + "/" + process_name + ".out";
    auto err_file = log_dir + "/" + process_name + ".err";
    return bp::child(target, bp::args(cmd),
                  bp::std_out > out_file,
                  bp::std_err > err_file,
                  bp::start_dir(cwd));
}

template <typename F>
void
wait_for(F &&f, const std::string& label = "", std::uint32_t timeout_sec=5)
{
    auto max_counts = timeout_sec * 10;
    for (int count = 0; count < max_counts; count++)
    {
        if (f())
        {
            return;
        }
        std::this_thread::sleep_for(std::chrono::milliseconds(100));
    }
    std::string err_msg = "Timeout";
    if (!label.empty())
    {
        err_msg += " (" + label + ")";
    }
    throw std::runtime_error(err_msg.c_str( ));
}

auto
file_exists( std::string full_path ) -> auto {
    return [full_path = std::move(full_path)]() -> bool {
        try {
            auto status = boost::filesystem::status(full_path);
            return boost::filesystem::exists(status);
        } catch(...) {
            return false;
        }
    };
}

auto
http_iface_up(std::uint16_t port) -> auto
{
    std::ostringstream os{};
    os << "http://127.0.0.1:" << port << "/";
    auto address = os.str();
    std::cerr << "waiting for http iface on " << address << std::endl;
    return [address]() -> bool {
        return ping_url(address);
    };
}

auto
has_channels(std::shared_ptr<daqd_stream::client> client) -> auto
{
    return [client = std::move(client)]() -> bool {
        return !client->channels(  ).empty( );
    };
}

TEST_CASE( "daqd_stream server can consume data from shared memory segments and serve its data" )
{
    auto address = [](const std::string& host, std::uint16_t port) -> std::string {
        std::ostringstream os{};
        os << host << ":" << port;
        return os.str( );
    };

    std::atomic< bool >  done_{ false };
    frame_utils::TempDir tmp_dir{ };
    auto test_root = tmp_dir.get( ).string( );
    auto master_file = test_root + "/master";

    std::string local_dc = helpers::random_mbuf_name(  );
    auto admin_port = helpers::random_port_number( );
    auto meta_port = helpers::random_port_number( );

    auto admin_address = address( "127.0.0.1", admin_port);
    auto meta_address = address( "127.0.0.1", meta_port);

    std::cerr << "Test root = " << test_root << std::endl;
    auto sim_stream = start_process("fe_simulated_streams", {"-R", "5", "-admin", admin_address, "-b", std::string(local_dc), "-i", std::string(test_root), "-s", "mbuf://"}, std::string(test_root), std::string(test_root));
    wait_for(file_exists(master_file), "ini files");
    wait_for(http_iface_up( admin_port ), "admin port");
    auto metadata_server = start_process("cds_metadata_server", {"-listen", meta_address, "-master", std::string(master_file)}, std::string(test_root), std::string(test_root));
    wait_for(http_iface_up( meta_port ), "metadata server");

    std::string conn_str =
        std::string( "daqm://") + local_dc + std::string("^meta://cds://" ) + meta_address;
    std::cerr << "Using connection string " << conn_str << std::endl;
    daqd_stream::buffer_parameters params{ };
    params.seconds_in_buffer = 4;
    params.total_buffer_size = 500 * 1024 * 1024;

    auto buffer_name = helpers::random_shm_name( );

    daqd_stream::daqd_stream server( conn_str, buffer_name, params, { } );
    {
        std::shared_ptr<daqd_stream::client> client{server.get_client(  )};

        std::thread server_loop_thread( [ &server, client ]( ) { server.run( ); } );
        {
            helpers::cleanup_action stop( [ &server ]( ) { server.stop( ); } );
            helpers::cleanup_action rm_shm(
                helpers::cleanup_shmem_action( { buffer_name } ) );

            wait_for( has_channels( client ), "channel list", 3000000000);
            std::cerr << "got " << client->channels( ).size( ) << " channels\n";

            auto channel_names = pick_channels( client->channels( ) );

            auto plan = client->plan_request(
                daqd_stream::PLAN_TYPE::PLAN_16TH, channel_names, 0 );
            auto plan_sec = client->plan_request(
                daqd_stream::PLAN_TYPE::PLAN_SEC, channel_names, 1 );
            std::vector< char > buffer( plan.required_size( ) );
            std::vector< char > buffer_sec( plan_sec.required_size( ) );

            daqd_stream::time_point      last_seen{ };
            daqd_stream::data_status     status{ };
            daqd_stream::sec_data_status status_sec{ };
            // std::cerr << "about to get data\n";

            auto expected_gps = 1000000000;
            auto expected_cycle = 15;

            std::uint64_t sec_requested = 0;

            do
            {
                client->get_16th_data(
                        plan, last_seen, buffer.data( ), buffer.size( ), status );
                last_seen = status.gps;
            } while (status.cycle != 14);

            expected_gps = status.gps.seconds;
            expected_cycle = 15;

            for ( auto i = 0; i < 32; ++i )
            {
                client->get_16th_data(
                    plan, last_seen, buffer.data( ), buffer.size( ), status );
                //            std::cerr << "got data: " << status.data_good << " "
                //                      << status.gps.seconds << ":" << status.cycle
                //                      << "\n";
                last_seen = status.gps;
                REQUIRE( status.data_good );
                REQUIRE( status.gps.seconds == expected_gps );
                REQUIRE( status.cycle == expected_cycle );
                if ( status.cycle == 15 )
                {
                    client->get_sec_data( plan_sec,
                                          sec_requested,
                                          1,
                                          buffer_sec.data( ),
                                          buffer_sec.size( ),
                                          status_sec );
                    REQUIRE( status_sec.gps.seconds == expected_gps );
                }
                expected_cycle = ( expected_cycle + 1 ) % 16;
                if ( expected_cycle == 0 )
                {
                    ++expected_gps;
                }
            }
        }
        client.reset();
        server_loop_thread.join( );
    }

}
