//
// simple_pv - a simple wrapper around the pCas libraries.
//
// Copyright (C) 2025 California Institute of Technology
//
// simple_pv is free software; you may redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 (GPLv3) of the
// License or at your discretion, any later version.
//
// LDASTools frameCPP is distributed in the hope that it will be useful, but
// without any warranty or even the implied warranty of merchantability
// or fitness for a particular purpose. See the GNU General Public
// License (GPLv2) for more details.
//
// Neither the names of the California Institute of Technology (Caltech),
// The Massachusetts Institute of Technology (M.I.T), The Laser
// Interferometer Gravitational-Wave Observatory (LIGO), nor the names
// of its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// You should have received a copy of the licensing terms for this
// software included in the file COPYING-GPL-3 located in the top-level
// directory of this package. If you did not, you can view a copy at
// http://dcc.ligo.org/M1500244/LICENSE
//


#ifndef SHAREABLE_SERVER_HH
#define SHAREABLE_SERVER_HH

#include <simple_pv/simple_epics.hh>
#include <simple_pv/shareable_server.hh>
#include <memory>
#include <vector>

namespace simple_epics
{
    /**
     * A ShareableServer give an interface to allow sharing a
     * simple_epics::Server between threads. Each instance of ShareableServer
     * tracks which PVs where created through it,and will destroy them when it
     * is destroyed.
     *
     * To create a new ShareableServer off of the same Server backend, call the
     * share() method which will return a new shared_ptr<SharedServer>.  The new
     * instance will use the sample simple_epics::Server as a backend, but will
     * have an empty local PV list.
     *
     * PVs should be created as normal, ie via calls to addPV.  But the addPV
     * calls should be done with the ShareableServer instance that matches it's
     * lifetime
     *
     * ShareableServer is not strictly thread-safe.  Though most calls are
     * simple pass throughs to a simple_epics::Server which is.
     */
    class ShareableServer
    {
        struct PrivateBuildInfo
        {
            std::shared_ptr< Server > server_{ nullptr };
        };

    public:
        explicit ShareableServer( std::string const& name )
            : instance_pvs_{ }, server_{ std::make_shared< Server >(
                                    name ) } { };
        explicit ShareableServer( PrivateBuildInfo builder )
            : instance_pvs_{ }, server_{ std::move( builder.server_ ) } { };
        ShareableServer( ) = delete;
        ShareableServer( const ShareableServer& ) = delete;
        ShareableServer( ShareableServer&& ) = delete;
        ShareableServer& operator=( const ShareableServer& ) = delete;
        ShareableServer& operator=( ShareableServer&& ) = delete;

        virtual ~ShareableServer( )
        {
            for ( const auto& pv : instance_pvs_ )
            {
                server_->destroyPV( pv.name );
            }
        }

        std::shared_ptr< ShareableServer >
        share( ) const
        {
            PrivateBuildInfo builder{ };
            builder.server_ = server_;
            return std::make_shared< ShareableServer >( std::move( builder ) );
        }

        /*!
         * @brief Add a PV to the server.
         */
        void
        addPV( pvUShortAttributes attr )
        {
            PVInfo info{ };
            info.name = attr.name( );
            info.pv_type = PVTypes::UInt16;
            info.mode = attr.mode( );
            server_->addPV( std::move( attr ) );
            instance_pvs_.emplace_back( info );
        }

        void
        addPV( pvIntAttributes attr )
        {
            PVInfo info{ };
            info.name = attr.name( );
            info.pv_type = PVTypes::Int32;
            info.mode = attr.mode( );
            server_->addPV( std::move( attr ) );
            instance_pvs_.emplace_back( info );
        }

        void
        addPV( pvUIntAttributes attr )
        {
            PVInfo info{ };
            info.name = attr.name( );
            info.pv_type = PVTypes::UInt32;
            info.mode = attr.mode( );
            server_->addPV( std::move( attr ) );
            instance_pvs_.emplace_back( info );
        }
        void
        addPV( pvStringAttributes attr )
        {
            PVInfo info{ };
            info.name = attr.name( );
            info.pv_type = PVTypes::String;
            info.mode = attr.mode( );
            server_->addPV( std::move( attr ) );
            instance_pvs_.emplace_back( info );
        }
        void
        addPV( pvFloatAttributes attr )
        {
            PVInfo info{ };
            info.name = attr.name( );
            info.pv_type = PVTypes::Float32;
            info.mode = attr.mode( );
            server_->addPV( std::move( attr ) );
            instance_pvs_.emplace_back( info );
        }
        void
        addPV( pvDoubleAttributes attr )
        {
            PVInfo info{ };
            info.name = attr.name( );
            info.pv_type = PVTypes::Float64;
            info.mode = attr.mode( );
            server_->addPV( std::move( attr ) );
            instance_pvs_.emplace_back( info );
        }

        /*!
         * @brief Reflect all changes in the data for each PV into the server
         * @note updates are done for all pv's in the shared server
         */
        void
        update( )
        {
            server_->update( );
        }

        const std::string&
        prefix( ) const noexcept
        {
            return server_->prefix( );
        }

        std::vector< PVInfo >
        active_pvs( ) const
        {
            std::vector< PVInfo > result{ };
            result.reserve( instance_pvs_.size( ) );

            std::string server_prefix = prefix( );

            for ( const auto& pv : instance_pvs_ )
            {
                PVInfo info{ pv };
                info.name = server_prefix + info.name;
                result.emplace_back( info );
            }
            return result;
        }

    private:
        std::vector< PVInfo >     instance_pvs_;
        std::shared_ptr< Server > server_;
    };
} // namespace simple_epics

#endif // SHAREABLE_SERVER_HH
