//
// simple_pv - a simple wrapper around the pCas libraries.
//
// Copyright (C) 2025 California Institute of Technology
//
// simple_pv is free software; you may redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 (GPLv3) of the
// License or at your discretion, any later version.
//
// LDASTools frameCPP is distributed in the hope that it will be useful, but
// without any warranty or even the implied warranty of merchantability
// or fitness for a particular purpose. See the GNU General Public
// License (GPLv2) for more details.
//
// Neither the names of the California Institute of Technology (Caltech),
// The Massachusetts Institute of Technology (M.I.T), The Laser
// Interferometer Gravitational-Wave Observatory (LIGO), nor the names
// of its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// You should have received a copy of the licensing terms for this
// software included in the file COPYING-GPL-3 located in the top-level
// directory of this package. If you did not, you can view a copy at
// http://dcc.ligo.org/M1500244/LICENSE
//

#ifndef DAQD_TRUNK_SIMPLE_EPICS_HH
#define DAQD_TRUNK_SIMPLE_EPICS_HH

#include <atomic>
#include <map>
#include <memory>
#include <mutex>
#include <string>
#include <type_traits>
#include <vector>
#include <gddAppFuncTable.h>
#include <epicsTimer.h>

#include <casdef.h>
#include <gddApps.h>
#include <smartGDDPointer.h>

#include <simple_pv/simple_pv.h>

namespace simple_epics
{
    enum class PVMode
    {
        ReadOnly = 0,
        ReadWrite = 1,
    };

    enum PVTypes
    {
        UInt16,
        Int32,
        UInt32,
        String,
        Float32,
        Float64,
        Unknown,
    };

    struct PVInfo
    {
        std::string  name{ };
        PVMode       mode{ PVMode::ReadOnly };
        enum PVTypes pv_type
        {
            PVTypes::Int32
        };
    };

    namespace detail
    {
        class simplePVBase : public casPV
        {
        public:
            simplePVBase( ) : casPV( )
            {
            }
            ~simplePVBase( ) override = default;

            virtual void update( ) = 0;

            virtual PVInfo info( ) const = 0;
        };
    } // namespace detail

    class Server;

    /*!
     * @brief A description of a PV, used to describe an int PV to the server.
     * @note this is given a pointer to the data.  This value is only read
     * when a Server object is told to update its data.
     */
    template < typename NumType >
    class pvBasicNumericAttributes
    {
        static_assert( std::is_integral< NumType >::value ||
                           std::is_floating_point< NumType >::value,
                       "integer or floating point type required" );

    public:
        using value_type = NumType;

        pvBasicNumericAttributes( std::string                   pv_name,
                                  std::atomic< NumType >*       value,
                                  std::pair< NumType, NumType > alarm_range,
                                  std::pair< NumType, NumType > warn_range,
                                  PVMode mode = PVMode::ReadOnly )
            : name_{ std::move( pv_name ) },

              alarm_low_{ alarm_range.first },
              alarm_high_{ alarm_range.second }, warn_low_{ warn_range.first },
              warn_high_{ warn_range.second }, mode_{ mode }, src_{ value }
        {
        }

        const std::string&
        name( ) const noexcept
        {
            return name_;
        }

        NumType
        alarm_high( ) const noexcept
        {
            return alarm_high_;
        }
        NumType
        alarm_low( ) const noexcept
        {
            return alarm_low_;
        }
        NumType
        warn_high( ) const noexcept
        {
            return warn_high_;
        }
        NumType
        warn_low( ) const noexcept
        {
            return warn_low_;
        }

        const std::atomic< NumType >*
        src( ) const noexcept
        {
            return src_;
        }

        PVMode
        mode( ) const noexcept
        {
            return mode_;
        }

    private:
        std::string name_;

        NumType alarm_high_;
        NumType alarm_low_;
        NumType warn_high_;
        NumType warn_low_;

        PVMode                  mode_;
        std::atomic< NumType >* src_{ nullptr };
    };
    using pvIntAttributes = pvBasicNumericAttributes< std::int32_t >;
    using pvUIntAttributes = pvBasicNumericAttributes< std::uint32_t >;
    using pvUShortAttributes = pvBasicNumericAttributes< std::uint16_t >;
    using pvFloatAttributes = pvBasicNumericAttributes< float >;
    using pvDoubleAttributes = pvBasicNumericAttributes< double >;
    static_assert( sizeof( std::int32_t ) == sizeof( int ),
                   "int must be 32 bit" );

    // pvStringData holds the data for a StringPV.
    // This is a non-copyable, non-moveable type as it contains
    // a mutex which is used to keep access safe.
    //
    // It should be used via a shared pointer.
    class pvStringData
    {
    public:
        pvStringData( ) = default;
        ~pvStringData( ) = default;
        pvStringData( const pvStringData& ) = delete;
        pvStringData& operator=( const pvStringData& ) = delete;
        pvStringData( pvStringData&& ) = delete;
        pvStringData& operator=( pvStringData&& ) = delete;

        std::string
        get( ) const
        {
            std::lock_guard< std::mutex > l_{ mutex_ };
            return value_;
        }

        void
        set( const std::string& s )
        {
            std::lock_guard< std::mutex > l_{ mutex_ };
            value_ = s;
        }

        explicit operator std::string( ) const
        {
            return get( );
        }

        pvStringData&
        operator=( const char* s )
        {
            s = ( s ? s : "" );
            std::lock_guard< std::mutex > l_{ mutex_ };
            value_ = s;
            return *this;
        }

        pvStringData&
        operator=( const std::string& s )
        {
            set( s );
            return *this;
        }

        pvStringData&
        operator=( std::string&& s )
        {
            std::lock_guard< std::mutex > l_{ mutex_ };
            value_ = s;
            return *this;
        }

    private:
        mutable std::mutex mutex_{ };
        std::string        value_{ };
    };

    /*!
     * @brief A description of a PV, used to describe a string PV to the server.
     * @note this is given a pointer to the data.  This value is only read
     * when a Server object is told to update its data.
     *
     * This version is used to support the clunky C api, do not use otherwise.
     */
    class pvStringAttributes
    {
    public:
        pvStringAttributes( std::string   pv_name,
                            pvStringData* data,
                            PVMode        mode = PVMode::ReadOnly )
            : name_{ std::move( pv_name ) }, mode_{ mode }, data_{ data }
        {
        }

        pvStringAttributes( std::string pv_name, char* data );

        const std::string&
        name( ) const noexcept
        {
            return name_;
        }

        PVMode
        mode( ) const noexcept
        {
            return mode_;
        }

        const pvStringData&
        value( ) const noexcept
        {
            return *data_;
        }

        pvStringData&
        value( ) noexcept
        {
            return *data_;
        }

    private:
        std::string   name_;
        PVMode        mode_;
        pvStringData* data_;
    };

    class pvCStringAttributes;

    /*!
     * @brief An R/O implementation of the Portable CA Server.
     */
    class Server : public caServer
    {
    public:
        explicit Server( std::string prefix = "" )
            : caServer( ), pvs_{ }, prefix_{ std::move( prefix ) }
        {
        }
        ~Server( ) override;

        /*!
         * @brief Add a PV to the server.
         */
        void addPV( pvUShortAttributes attr );
        void addPV( pvIntAttributes attr );
        void addPV( pvUIntAttributes attr );
        void addPV( pvCStringAttributes attr );
        void addPV( pvStringAttributes attr );
        void addPV( pvFloatAttributes attr );
        void addPV( pvDoubleAttributes attr );

        /*!
         * Remove a PV from the server.
         * @param name PV name (without prefix) to remove from the server
         * @return True if it was removed, false if it could not be found.
         */
        bool destroyPV( const std::string& name );

        const std::string&
        prefix( ) const noexcept
        {
            return prefix_;
        }

        /*!
         * @brief Reflect all changes in the data for each PV into the server
         */
        void update( );

        pvExistReturn pvExistTest( const casCtx&    ctx,
                                   const caNetAddr& clientAddress,
                                   const char*      pPVAliasName ) override;

        pvAttachReturn pvAttach( const casCtx& ctx,
                                 const char*   pPVAliasName ) override;

        std::vector< PVInfo > active_pvs( ) const;

    private:
        template < typename T >
        std::string
        attribute_name( const T& attribute ) const
        {
            return prefix_ + attribute.name( );
        }

        std::mutex                                                       m_;
        std::map< std::string, std::unique_ptr< detail::simplePVBase > > pvs_;
        std::string prefix_;
    };

} // namespace simple_epics

#endif // DAQD_TRUNK_SIMPLE_EPICS_HH
